/***************************************************************************
 *  GModelSpatialEllipticalGauss.cpp - Elliptical gauss source model class *
 * ----------------------------------------------------------------------- *
 *  copyright (C) 2015-2024 by Michael Mayer                               *
 * ----------------------------------------------------------------------- *
 *                                                                         *
 *  This program is free software: you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, either version 3 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 *                                                                         *
 ***************************************************************************/
/**
 * @file GModelSpatialEllipticalGauss.cpp
 * @brief Elliptical gauss model class implementation
 * @author Michael Mayer
 */

/* __ Includes ___________________________________________________________ */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include "GException.hpp"
#include "GTools.hpp"
#include "GMath.hpp"
#include "GModelSpatialEllipticalGauss.hpp"
#include "GModelSpatialRadialDisk.hpp"
#include "GModelSpatialRegistry.hpp"

/* __ Constants __________________________________________________________ */
namespace {
    const double c_theta_max    = 5.0; //!< semiaxis multiplied for theta_max()
    const double c_max_exponent = 0.5 * c_theta_max * c_theta_max;
    const double c_fraction     = 1.0 - std::exp(-c_max_exponent);
}

/* __ Globals ____________________________________________________________ */
const GModelSpatialEllipticalGauss g_elliptical_gauss_seed;
const GModelSpatialRegistry        g_elliptical_gauss_registry(&g_elliptical_gauss_seed);
#if defined(G_LEGACY_XML_FORMAT)
const GModelSpatialEllipticalGauss g_elliptical_gauss_legacy_seed(true, "EllipticalGauss");
const GModelSpatialRegistry        g_elliptical_gauss_legacy_registry(&g_elliptical_gauss_legacy_seed);
#endif

/* __ Method name definitions ____________________________________________ */
#define G_CONSTRUCTOR                        "GModelSpatialEllipticalGauss::"\
                  "GModelSpatialEllipticalGauss(GSkyDir&, double&, double&, "\
                                                     "double&, std::string&)"
#define G_READ             "GModelSpatialEllipticalGauss::read(GXmlElement&)"
#define G_WRITE           "GModelSpatialEllipticalGauss::write(GXmlElement&)"

/* __ Macros _____________________________________________________________ */

/* __ Coding definitions _________________________________________________ */

/* __ Debug definitions __________________________________________________ */


/*==========================================================================
 =                                                                         =
 =                        Constructors/destructors                         =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Void constructor
 ***************************************************************************/
GModelSpatialEllipticalGauss::GModelSpatialEllipticalGauss(void) :
                              GModelSpatialElliptical()
{
    // Initialise members
    init_members();

    // Return
    return;
}


/***********************************************************************//**
 * @brief Model type constructor
 *
 * @param[in] dummy Dummy flag.
 * @param[in] type Model type.
 *
 * Constructs empty elliptical Gaussian model by specifying a model @p type.
 ***************************************************************************/
GModelSpatialEllipticalGauss::GModelSpatialEllipticalGauss(const bool&        dummy,
                                                           const std::string& type) :
                              GModelSpatialElliptical()
{
    // Initialise members
    init_members();

    // Set model type
    m_type = type;

    // Return
    return;
}


/***********************************************************************//**
 * @brief Elliptical Gaussian constructor
 *
 * @param[in] dir Centre of elliptical Gaussian.
 * @param[in] semimajor Semi-major axis (degrees).
 * @param[in] semiminor Semi-minor axis (degrees).
 * @param[in] posangle Position angle of semi-major axis (degrees).
 * @param[in] coordsys Coordinate system (either "CEL" or "GAL")
 *
 * @exception GException::invalid_argument
 *            Invalid @p coordsys argument specified.
 *
 * Construct an elliptical Gaussian model from the ellipse centre direction
 * (@p dir), the @p semimajor and @p semiminor axes, and the position
 * angle (@p posangle). The @p coordsys parameter specifies whether the sky
 * direction and the position angle should be interpreted in the celestial
 * or Galactic coordinates.
 ***************************************************************************/
GModelSpatialEllipticalGauss::GModelSpatialEllipticalGauss(const GSkyDir&     dir,
                                                           const double&      semimajor,
                                                           const double&      semiminor,
                                                           const double&      posangle,
                                                           const std::string& coordsys) :
                              GModelSpatialElliptical()
{
    // Throw an exception if the coordinate system is invalid
    if ((coordsys != "CEL") && (coordsys != "GAL")) {
        std::string msg = "Invalid coordinate system \""+coordsys+"\" "
                          "specified. Please specify either \"CEL\" or "
                          "\"GAL\".";
        throw GException::invalid_argument(G_CONSTRUCTOR, msg);
    }

    // Initialise members
    init_members();

    // Set parameter names
    if (coordsys == "CEL") {
        m_lon.name("RA");
        m_lat.name("DEC");
    }
    else {
        m_lon.name("GLON");
        m_lat.name("GLAT");
    }

    // Assign parameters
    this->dir(dir);
    this->semiminor(semiminor);
    this->semimajor(semimajor);
    this->posangle(posangle);

    // Return
    return;
}


/***********************************************************************//**
 * @brief XML constructor
 *
 * @param[in] xml XML element.
 *
 * Constructs elliptical Gaussian model by extracting information from an XML
 * element. See the read() method for more information about the expected
 * structure of the XML element.
 ***************************************************************************/
GModelSpatialEllipticalGauss::GModelSpatialEllipticalGauss(const GXmlElement& xml) :
                              GModelSpatialElliptical()
{
    // Initialise members
    init_members();

    // Read information from XML element
    read(xml);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Copy constructor
 *
 * @param[in] model Elliptical Gaussian model.
 ***************************************************************************/
GModelSpatialEllipticalGauss::GModelSpatialEllipticalGauss(const GModelSpatialEllipticalGauss& model) :
                              GModelSpatialElliptical(model)
{
    // Initialise members
    init_members();

    // Copy members
    copy_members(model);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Destructor
 ***************************************************************************/
GModelSpatialEllipticalGauss::~GModelSpatialEllipticalGauss(void)
{
    // Free members
    free_members();

    // Return
    return;
}


/*==========================================================================
 =                                                                         =
 =                               Operators                                 =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Assignment operator
 *
 * @param[in] model Elliptical gauss model.
 * @return Elliptical Gaussian model.
 ***************************************************************************/
GModelSpatialEllipticalGauss& GModelSpatialEllipticalGauss::operator=(const GModelSpatialEllipticalGauss& model)
{
    // Execute only if object is not identical
    if (this != &model) {

        // Copy base class members
        this->GModelSpatialElliptical::operator=(model);

        // Free members
        free_members();

        // Initialise members
        init_members();

        // Copy members
        copy_members(model);

    } // endif: object was not identical

    // Return
    return *this;
}


/*==========================================================================
 =                                                                         =
 =                            Public methods                               =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Clear elliptical Gaussian model
 ***************************************************************************/
void GModelSpatialEllipticalGauss::clear(void)
{
    // Free class members (base and derived classes, derived class first)
    free_members();
    this->GModelSpatialElliptical::free_members();
    this->GModelSpatial::free_members();

    // Initialise members
    this->GModelSpatial::init_members();
    this->GModelSpatialElliptical::init_members();
    init_members();

    // Return
    return;
}


/***********************************************************************//**
 * @brief Clone elliptical Gaussian model
 *
 * @return Pointer to deep copy of elliptical Gaussian model.
 ***************************************************************************/
GModelSpatialEllipticalGauss* GModelSpatialEllipticalGauss::clone(void) const
{
    // Clone elliptical Gaussian model
    return new GModelSpatialEllipticalGauss(*this);
}


/***********************************************************************//**
 * @brief Evaluate function (in units of sr\f$^{-1}\f$)
 *
 * @param[in] theta Angular distance from disk centre (radians)
 * @param[in] posangle Position angle (counterclockwise from North) (radians)
 * @param[in] energy Photon energy (not used)
 * @param[in] time Photon arrival time (not used)
 * @param[in] gradients Compute gradients? (not used)
 * @return Model value (sr\f$^{-1}\f$)
 *
 * Evaluates the spatial component for an elliptical Gaussian source model.
 *
 * The elliptical Gaussian function is energy and time independent, and
 * defined by
 *
 * \f[
 *    S_{\rm p}(\theta, \phi | E, t) = S_{\rm p}(\theta, \phi)
 * \f]
 *
 * where \f$\theta\f$ is the angular separation from the centre of the
 * elliptical Gaussian and \f$\phi\f$ the position angle with respect to
 * the model centre, counted counterclockwise from North. The coordinate
 * system in which the position angle \f$\phi\f$ is given is defined by the
 * coordinate system in which the centre of elliptical Gaussian model is
 * specified. The coordsys() method may be used to retrieve a string of the
 * coordinate system.
 *
 * The function \f$S_{\rm p}(\theta, \phi)\f$ is given by
 *
 * \f[
 *    S_{\rm p}(\theta, \phi) = {\tt m\_norm} \times
 *    \exp \left( -\frac{\theta^2}{2} \left[
 *    \left( \frac{\sin^2 \phi_0}{b^2} + \frac{\cos^2 \phi_0}{a^2} \right) \cos^2 \phi +
 *    \left( \frac{\sin^2 \phi_0}{a^2} + \frac{\cos^2 \phi_0}{b^2} \right) \sin^2 \phi +
 *    \left( \frac{\sin 2\phi_0}{a^2} - \frac{\sin 2\phi_0}{b^2} \right) \sin \phi \cos \phi
 *    \right] \right)
 * \f]
 *
 * where
 * \f$a\f$ is the semi-major axis of the ellipse,
 * \f$b\f$ is the semi-minor axis, and
 * \f$\phi_0\f$ is the position angle of the ellipse, counted counterclockwise
 * from North.
 *
 * \f${\tt m\_norm}\f$ is a normalization constant that is approximated by
 *
 * \f[
 *    {\tt m\_norm} = \frac{1}{2 \pi \times a \times b}
 * \f]
 *
 * which is only accurate for values of \f$a\f$ and \f$b\f$ that do not
 * exceed a few degrees. For larger ellipse sizes it is recommended to
 * spatially integrate \f$S_{\rm p}(\theta, \phi)\f$ in order to obtain a
 * proper normalisation.
 *
 * The method will not compute analytical parameter gradients, even if the
 * @p gradients argument is set to true. Elliptical Gaussian parameter
 * gradients need to be computed numerically.
 ***************************************************************************/
double GModelSpatialEllipticalGauss::eval(const double&  theta,
                                          const double&  posangle,
                                          const GEnergy& energy,
                                          const GTime&   time,
                                          const bool&    gradients) const
{
    // Initialise value
    double value = 0.0;

    // Continue only if we're inside circle enclosing the ellipse
    if (theta <= theta_max()) {

        // Update precomputation cache
        update();

        // Compute exponent
        double sinphi   = std::sin(posangle);
        double cosphi   = std::cos(posangle);
        double term1    = m_term1 * cosphi * cosphi;
        double term2    = m_term2 * sinphi * sinphi;
        double term3    = m_term3 * sinphi * cosphi;
        double exponent = theta * theta * (term1 + term2 + term3);

        // Compute value
        value = m_norm * std::exp(-exponent);

        // Compile option: Check for NaN/Inf
        #if defined(G_NAN_CHECK)
        if (gammalib::is_notanumber(value) || gammalib::is_infinite(value)) {
            std::cout << "*** ERROR: GModelSpatialEllipticalGauss::eval";
            std::cout << "(theta=" << theta << "): NaN/Inf encountered";
            std::cout << "(posangle=" << posangle << "): NaN/Inf encountered";
            std::cout << " (value=" << value;
            std::cout << ", m_term1=" << m_term1;
            std::cout << ", m_term2=" << m_term2;
            std::cout << ", m_term3=" << m_term3;
            std::cout << ", m_norm=" << m_norm;
            std::cout << ")" << std::endl;
        }
        #endif

    } // endif: position was inside enclosing circle

    // Return value
    return value;
}


/***********************************************************************//**
 * @brief Returns MC sky direction
 *
 * @param[in] energy Photon energy.
 * @param[in] time Photon arrival time.
 * @param[in,out] ran Random number generator.
 * @return Sky direction.
 *
 * Draws an arbitrary sky direction from the elliptical Gaussian model.
 *
 * @warning
 * For numerical reasons the elliptical Gaussian will be truncated for
 * \f$\theta\f$ angles larger than 5 times the effective ellipse
 * radius.
 *
 * @warning
 * The method uses a small angle approximation.
 ***************************************************************************/
GSkyDir GModelSpatialEllipticalGauss::mc(const GEnergy& energy,
                                         const GTime&   time,
                                         GRan&          ran) const
{
    // Update precomputation cache
    update();

    // Initialise photon
    GPhoton photon;
    photon.energy(energy);
    photon.time(time);

    // Draw gaussian offset from each axis
    double ran_major;
    double ran_minor;
    do {
        ran_major = ran.normal();
    } while (ran_major > c_theta_max);
    do {
        ran_minor = ran.normal();
    } while (ran_minor > c_theta_max);
    double theta1 = semimajor() * ran_major;
    double theta2 = semiminor() * ran_minor;

    // Compute total offset from model centre in small angle approximation
    double theta = std::sqrt(theta1 * theta1 + theta2 * theta2);

    // Compute rotation angle, taking into account given position angle
    double phi = gammalib::atan2d(theta2, theta1) + posangle();

    // Rotate sky direction by offset
    GSkyDir sky_dir = dir();
    sky_dir.rotate_deg(phi , theta);

    // Set photon sky direction
    photon.dir(sky_dir);

    // Return photon direction
    return (photon.dir());
}


/***********************************************************************//**
 * @brief Checks whether model contains specified sky direction
 *
 * @param[in] dir Sky direction.
 * @param[in] margin Margin to be added to sky direction (degrees)
 * @return True if the model contains the sky direction.
 *
 * Signals whether a sky direction is contained in the elliptical gauss
 * model.
 *
 * @todo Implement correct evaluation of effective ellipse radius.
 ***************************************************************************/
bool GModelSpatialEllipticalGauss::contains(const GSkyDir& dir,
                                            const double&  margin) const
{
    // Compute distance to centre (radian)
    double distance = dir.dist(this->dir());

    // Return flag
    return (distance <= theta_max() + margin*gammalib::deg2rad);
}


/***********************************************************************//**
 * @brief Return maximum model radius (in radians)
 *
 * @return Returns maximum model radius.
 *
 * Returns the maximum model radius which is defined as the maximum of
 * 5 semimajor() and 5 semiminor().
 ***************************************************************************/
double GModelSpatialEllipticalGauss::theta_max(void) const
{
    // Set maximum model radius
    double theta_max = (semimajor() > semiminor())
                       ? semimajor() * gammalib::deg2rad * c_theta_max
                       : semiminor() * gammalib::deg2rad * c_theta_max;

    // Return value
    return theta_max;
}


/***********************************************************************//**
 * @brief Read model from XML element
 *
 * @param[in] xml XML element.
 *
 * Reads the elliptical gauss model information from an XML element. The XML
 * element shall have either the format 
 *
 *     <spatialModel type="EllipticalGaussian">
 *       <parameter name="RA"          scale="1.0" value="83.6331" min="-360" max="360" free="1"/>
 *       <parameter name="DEC"         scale="1.0" value="22.0145" min="-90"  max="90"  free="1"/>
 *       <parameter name="PA"          scale="1.0" value="45.0"    min="-360"  max="360" free="1"/>
 *       <parameter name="MinorRadius" scale="1.0" value="0.5"     min="0.001" max="10"  free="1"/>
 *       <parameter name="MajorRadius" scale="1.0" value="2.0"     min="0.001" max="10"  free="1"/>
 *     </spatialModel>
 *
 * or
 *
 *     <spatialModel type="EllipticalGaussian">
 *       <parameter name="GLON"        scale="1.0" value="83.6331" min="-360" max="360" free="1"/>
 *       <parameter name="GLAT"        scale="1.0" value="22.0145" min="-90"  max="90"  free="1"/>
 *       <parameter name="PA"          scale="1.0" value="45.0"    min="-360"  max="360" free="1"/>
 *       <parameter name="MinorRadius" scale="1.0" value="0.5"     min="0.001" max="10"  free="1"/>
 *       <parameter name="MajorRadius" scale="1.0" value="2.0"     min="0.001" max="10"  free="1"/>
 *     </spatialModel>
 *
 * @todo Implement a test of the ellipse boundary. The axes
 *       and axes minimum should be >0.
 ***************************************************************************/
void GModelSpatialEllipticalGauss::read(const GXmlElement& xml)
{
    // Verify number of model parameters
    gammalib::xml_check_parnum(G_READ, xml, 5);

    // Read gauss location
    GModelSpatialElliptical::read(xml);

    // Get parameters
    const GXmlElement* minor = gammalib::xml_get_par(G_READ, xml, m_semiminor.name());
    const GXmlElement* major = gammalib::xml_get_par(G_READ, xml, m_semimajor.name());

    // Read parameters
    m_semiminor.read(*minor);
    m_semimajor.read(*major);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Write model into XML element
 *
 * @param[in] xml XML element into which model information is written.
 *
 * Write the elliptical gauss model information into an XML element. The XML
 * element will have the format 
 *
 *     <spatialModel type="EllipticalGaussian">
 *       <parameter name="RA"          scale="1.0" value="83.6331" min="-360" max="360" free="1"/>
 *       <parameter name="DEC"         scale="1.0" value="22.0145" min="-90"  max="90"  free="1"/>
 *       <parameter name="PA"          scale="1.0" value="45.0"    min="-360"  max="360" free="1"/>
 *       <parameter name="MinorRadius" scale="1.0" value="0.5"     min="0.001" max="10"  free="1"/>
 *       <parameter name="MajorRadius" scale="1.0" value="2.0"     min="0.001" max="10"  free="1"/>
 *     </spatialModel>
 *
 ***************************************************************************/
void GModelSpatialEllipticalGauss::write(GXmlElement& xml) const
{
    // Verify model type
    gammalib::xml_check_type(G_WRITE, xml, type());

    // Write disk location
    GModelSpatialElliptical::write(xml);

    // Get or create parameters
    GXmlElement* minor = gammalib::xml_need_par(G_WRITE, xml, m_semiminor.name());
    GXmlElement* major = gammalib::xml_need_par(G_WRITE, xml, m_semimajor.name());

    // Write parameters
    m_semiminor.write(*minor);
    m_semimajor.write(*major);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Print information
 *
 * @param[in] chatter Chattiness.
 * @return String containing model information.
 ***************************************************************************/
std::string GModelSpatialEllipticalGauss::print(const GChatter& chatter) const
{
    // Initialise result string
    std::string result;

    // Continue only if chatter is not silent
    if (chatter != SILENT) {

        // Append header
        result.append("=== GModelSpatialEllipticalGauss ===");

        // Append parameters
        result.append("\n"+gammalib::parformat("Number of parameters"));
        result.append(gammalib::str(size()));
        for (int i = 0; i < size(); ++i) {
            result.append("\n"+m_pars[i]->print(chatter));
        }

    } // endif: chatter was not silent

    // Return result
    return result;
}


/*==========================================================================
 =                                                                         =
 =                            Private methods                              =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Initialise class members
 ***************************************************************************/
void GModelSpatialEllipticalGauss::init_members(void)
{
    // Initialise model type
    m_type = "EllipticalGaussian";

    // Initialise precomputation cache. Note that zero values flag
    // uninitialised as a zero radius is not meaningful
    m_last_minor    = 0.0;
    m_last_major    = 0.0;
    m_minor_rad     = 0.0;
    m_major_rad     = 0.0;
    m_norm          = 0.0;
    m_last_posangle = 9999.0; // Signals that has not been initialised
    m_sin2pos       = 0.0;
    m_cospos2       = 0.0;
    m_sinpos2       = 0.0;
    m_minor2        = 0.0;
    m_major2        = 0.0;
    m_term1         = 0.0;
    m_term2         = 0.0;
    m_term3         = 0.0;

    // Return
    return;
}


/***********************************************************************//**
 * @brief Copy class members
 *
 * @param[in] model Elliptical Gaussian model.
 ***************************************************************************/
void GModelSpatialEllipticalGauss::copy_members(const GModelSpatialEllipticalGauss& model)
{
    // Copy members
    m_type = model.m_type;   // Needed to conserve model type

    // Copy precomputation cache
    m_last_minor    = model.m_last_minor;
    m_last_major    = model.m_last_major;
    m_minor_rad     = model.m_minor_rad;
    m_major_rad     = model.m_major_rad;
    m_norm          = model.m_norm;
    m_last_posangle = model.m_last_posangle;
    m_sin2pos       = model.m_sin2pos;
    m_cospos2       = model.m_cospos2;
    m_sinpos2       = model.m_sinpos2;
    m_minor2        = model.m_minor2;
    m_major2        = model.m_major2;
    m_term1         = model.m_term1;
    m_term2         = model.m_term2;
    m_term3         = model.m_term3;

    // Return
    return;
}


/***********************************************************************//**
 * @brief Delete class members
 ***************************************************************************/
void GModelSpatialEllipticalGauss::free_members(void)
{
    // Return
    return;
}


/***********************************************************************//**
 * @brief Update precomputation cache
 *
 * Precomputes several variables in case that the model parameters
 * semiminor(), semimajor() or posangle() changed. Precomputation speeds
 * up the model evaluation.
 *
 * The method also computes the normalization
 *
 * \f[
 *    {\tt m\_norm} = \frac{1}{2 \pi \times a \times b}
 * \f]
 *
 * where 
 * \f$a\f$ is the semimajor() axis and
 * \f$b\f$ is the semiminor() axis of the ellipse.
 *
 * @warning
 * The normalization of the elliptical Gaussian is only valid in the small
 * angle approximation.
 ***************************************************************************/
void GModelSpatialEllipticalGauss::update() const
{
    // Initialise flag if something has changed
    bool changed = false;

    // Update if one axis has changed
    if (m_last_minor != semiminor() || m_last_major != semimajor()) {

        // Signal parameter changes
        changed = true;

        // Store last values
        m_last_minor = semiminor();
        m_last_major = semimajor();

        // Compute axes in radians
        m_minor_rad = m_last_minor * gammalib::deg2rad;
        m_major_rad = m_last_major * gammalib::deg2rad;

        // Take their squares
        m_minor2 = m_minor_rad * m_minor_rad;
        m_major2 = m_major_rad * m_major_rad;

        // Compute normalisation.
        // TODO: Implement correct formula. This one is not correct on
        // TODO: a sphere, but it's fine for small Gaussians.
        double denom = gammalib::twopi * m_minor_rad * m_major_rad;
        m_norm       = (denom > 0.0) ? 1.0 / denom : 0.0;

    } // endif: update required

    // Update chache if position angle changed
    if (m_last_posangle != posangle()) {

        // Signal parameter changes
        changed = true;

        // Store last value
        m_last_posangle = posangle();

        // Compute angle in radians
        double posangle_rad = m_last_posangle * gammalib::deg2rad;

        // Compute sine and cosine
        double cospos = std::cos(posangle_rad);
        double sinpos = std::sin(posangle_rad);

        // Cache important values for further computations
        m_cospos2 = cospos * cospos;
        m_sinpos2 = sinpos * sinpos;
        m_sin2pos = std::sin(2.0 * posangle_rad);

    } // endif: position angle update required

    // Perform precomputations in case anything has changed
    if (changed) {

        // Compute terms needed to compute inverse effective radius
        // squared
        m_term1 = 0.5 * (m_sinpos2 / m_minor2 + m_cospos2 / m_major2);
        m_term2 = 0.5 * (m_sinpos2 / m_major2 + m_cospos2 / m_minor2);
        m_term3 = 0.5 * (m_sin2pos / m_major2 - m_sin2pos / m_minor2);

    } // endif: something has changed

    // Return
    return;
}


/***********************************************************************//**
 * @brief Set boundary sky region
 ***************************************************************************/
void GModelSpatialEllipticalGauss::set_region(void) const
{
    // Set maximum model radius
    double max_radius = (semimajor() > semiminor()) ? semimajor() : semiminor();

    // Set sky region circle
    GSkyRegionCircle region(dir(), max_radius * c_theta_max);

    // Set region (circumvent const correctness)
    const_cast<GModelSpatialEllipticalGauss*>(this)->m_region = region;

    // Return
    return;
}
