/***************************************************************************
 *   GModelSpatialEllipticalDisk.cpp - Elliptical disk source model class  *
 * ----------------------------------------------------------------------- *
 *  copyright (C) 2013-2024 by Michael Mayer                               *
 * ----------------------------------------------------------------------- *
 *                                                                         *
 *  This program is free software: you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, either version 3 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 *                                                                         *
 ***************************************************************************/
/**
 * @file GModelSpatialEllipticalDisk.cpp
 * @brief Elliptical disk model class implementation
 * @author Michael Mayer
 */

/* __ Includes ___________________________________________________________ */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include "GException.hpp"
#include "GTools.hpp"
#include "GMath.hpp"
#include "GModelSpatialEllipticalDisk.hpp"
#include "GModelSpatialRadialDisk.hpp"
#include "GModelSpatialRegistry.hpp"

/* __ Constants __________________________________________________________ */

/* __ Globals ____________________________________________________________ */
const GModelSpatialEllipticalDisk g_elliptical_disk_seed;
const GModelSpatialRegistry       g_elliptical_disk_registry(&g_elliptical_disk_seed);

/* __ Method name definitions ____________________________________________ */
#define G_CONSTRUCTOR                         "GModelSpatialEllipticalDisk::"\
                   "GModelSpatialEllipticalDisk(GSkyDir&, double&, double&, "\
                                                     "double&, std::string&)"
#define G_READ              "GModelSpatialEllipticalDisk::read(GXmlElement&)"
#define G_WRITE            "GModelSpatialEllipticalDisk::write(GXmlElement&)"

/* __ Macros _____________________________________________________________ */

/* __ Coding definitions _________________________________________________ */

/* __ Debug definitions __________________________________________________ */


/*==========================================================================
 =                                                                         =
 =                        Constructors/destructors                         =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Void constructor
 ***************************************************************************/
GModelSpatialEllipticalDisk::GModelSpatialEllipticalDisk(void) :
                             GModelSpatialElliptical()
{
    // Initialise members
    init_members();

    // Return
    return;
}


/***********************************************************************//**
 * @brief Disk constructor
 *
 * @param[in] dir Sky position of disk centre.
 * @param[in] semimajor Semi-major axis (degrees).
 * @param[in] semiminor Semi-minor axis (degrees).
 * @param[in] posangle Position angle of semi-major axis (degrees).
 * @param[in] coordsys Coordinate system (either "CEL" or "GAL")
 *
 * @exception GException::invalid_argument
 *            Invalid @p coordsys argument specified.
 *
 * Construct elliptical disk model from sky position of the ellipse centre
 * (@p dir), the @p semimajor and @p semiminor axes, and the position
 * angle (@p posangle). The @p coordsys parameter specifies whether the sky
 * direction and the position angle should be interpreted in the celestial
 * or Galactic coordinates.
 ***************************************************************************/
GModelSpatialEllipticalDisk::GModelSpatialEllipticalDisk(const GSkyDir&     dir,
                                                         const double&      semimajor,
                                                         const double&      semiminor,
                                                         const double&      posangle,
                                                         const std::string& coordsys) :
                             GModelSpatialElliptical()
{
    // Throw an exception if the coordinate system is invalid
    if ((coordsys != "CEL") && (coordsys != "GAL")) {
        std::string msg = "Invalid coordinate system \""+coordsys+"\" "
                          "specified. Please specify either \"CEL\" or "
                          "\"GAL\".";
        throw GException::invalid_argument(G_CONSTRUCTOR, msg);
    }

    // Initialise members
    init_members();

    // Set parameter names
    if (coordsys == "CEL") {
        m_lon.name("RA");
        m_lat.name("DEC");
    }
    else {
        m_lon.name("GLON");
        m_lat.name("GLAT");
    }

    // Assign parameters
    this->dir(dir);
    this->semiminor(semiminor);
    this->semimajor(semimajor);
    this->posangle(posangle);

    // Return
    return;
}


/***********************************************************************//**
 * @brief XML constructor
 *
 * @param[in] xml XML element.
 *
 * Constructs elliptical disk model by extracting information from an XML
 * element. See the read() method for more information about the expected
 * structure of the XML element.
 ***************************************************************************/
GModelSpatialEllipticalDisk::GModelSpatialEllipticalDisk(const GXmlElement& xml) :
                             GModelSpatialElliptical()
{
    // Initialise members
    init_members();

    // Read information from XML element
    read(xml);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Copy constructor
 *
 * @param[in] model Elliptical disk model.
 ***************************************************************************/
GModelSpatialEllipticalDisk::GModelSpatialEllipticalDisk(const GModelSpatialEllipticalDisk& model) :
                             GModelSpatialElliptical(model)
{
    // Initialise members
    init_members();

    // Copy members
    copy_members(model);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Destructor
 ***************************************************************************/
GModelSpatialEllipticalDisk::~GModelSpatialEllipticalDisk(void)
{
    // Free members
    free_members();

    // Return
    return;
}


/*==========================================================================
 =                                                                         =
 =                               Operators                                 =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Assignment operator
 *
 * @param[in] model Elliptical disk model.
 * @return Elliptical disk model.
 ***************************************************************************/
GModelSpatialEllipticalDisk& GModelSpatialEllipticalDisk::operator=(const GModelSpatialEllipticalDisk& model)
{
    // Execute only if object is not identical
    if (this != &model) {

        // Copy base class members
        this->GModelSpatialElliptical::operator=(model);

        // Free members
        free_members();

        // Initialise members
        init_members();

        // Copy members
        copy_members(model);

    } // endif: object was not identical

    // Return
    return *this;
}


/*==========================================================================
 =                                                                         =
 =                            Public methods                               =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Clear elliptical disk model
 ***************************************************************************/
void GModelSpatialEllipticalDisk::clear(void)
{
    // Free class members (base and derived classes, derived class first)
    free_members();
    this->GModelSpatialElliptical::free_members();
    this->GModelSpatial::free_members();

    // Initialise members
    this->GModelSpatial::init_members();
    this->GModelSpatialElliptical::init_members();
    init_members();

    // Return
    return;
}


/***********************************************************************//**
 * @brief Clone elliptical disk model
 *
 * @return Pointer to deep copy of elliptical disk model.
 ***************************************************************************/
GModelSpatialEllipticalDisk* GModelSpatialEllipticalDisk::clone(void) const
{
    // Clone elliptical disk model
    return new GModelSpatialEllipticalDisk(*this);
}


/***********************************************************************//**
 * @brief Evaluate function (in units of sr^-1)
 *
 * @param[in] theta Angular distance from disk centre (radians).
 * @param[in] posangle Position angle (counterclockwise from North) (radians).
 * @param[in] energy Photon energy.
 * @param[in] time Photon arrival time.
 * @param[in] gradients Compute gradients?
 * @return Model value.
 *
 * Evaluates the spatial component for an elliptical disk source model.
 *
 * The elliptical disk function is energy and time independent, and defined
 * by
 *
 * \f[
 *    S_{\rm p}(\theta, \phi | E, t) = S_{\rm p}(\theta, \phi)
 * \f]
 *
 * where \f$\theta\f$ is the angular separation from the centre of the
 * elliptical disk and \f$\phi\f$ the position angle with respect to
 * the model centre, counted counterclockwise from North. The coordinate
 * system in which the position angle \f$\phi\f$ is given is defined by the
 * coordinate system in which the centre of elliptical Gaussian model is
 * specified. The coordsys() method may be used to retrieve a string of the
 * coordinate system.
 *
 * \f$S_{\rm p}(\theta, \phi)\f$ is defined by
 *
 * \f[
 *    S_{\rm p}(\theta, \phi | E, t) = \left \{
 *    \begin{array}{l l}
 *     {\tt m\_norm}
 *     & \mbox{if} \, \, \theta \le \theta_{\rm eff}(\phi) \\
 *     \\
 *     0 & \mbox{else}
 *    \end{array}
 *    \right .
 * \f]
 *
 * where
 *
 * \f[
 *    \theta_{\rm eff}(\phi) = \left[
 *    \left( \frac{\sin^2 \phi_0}{b^2} + \frac{\cos^2 \phi_0}{a^2} \right) \cos^2 \phi +
 *    \left( \frac{\sin^2 \phi_0}{a^2} + \frac{\cos^2 \phi_0}{b^2} \right) \sin^2 \phi +
 *    \left( \frac{\sin 2\phi_0}{a^2} - \frac{\sin 2\phi_0}{b^2} \right) \sin \phi \cos \phi
 *    \right]^{-1/2}
 * \f]
 *
 * is the effective radius of the elliptical disk for a position angle
 * of \f$\phi\f$, \f$a\f$ is the semi-major axis of the ellipse, and
 * \f$b\f$ is the semi-minor axis.
 *
 * The normalisation constant \f${\tt m\_norm}\f$ is given by
 *
 * \f[
 *    {\tt m\_norm} = \frac{1}{2 \pi \sqrt{(1-\cos a)(1-\cos b})}
 * \f]
 *
 * The method will not compute analytical parameter gradients, even if the
 * @p gradients argument is set to true. Radial disk parameter gradients
 * need to be computed numerically.
 ***************************************************************************/
double GModelSpatialEllipticalDisk::eval(const double&  theta,
                                         const double&  posangle,
                                         const GEnergy& energy,
                                         const GTime&   time,
                                         const bool&    gradients) const
{
    // Initialise value
    double value = 0.0;

    // Continue only if we're inside circle enclosing the ellipse
    if (theta <= theta_max()) {

        // Update precomputation cache
        update();

        // Compute theta0
        double sinphi = std::sin(posangle);
        double cosphi = std::cos(posangle);
        double term1  = m_term1 * cosphi * cosphi;
        double term2  = m_term2 * sinphi * sinphi;
        double term3  = m_term3 * sinphi * cosphi;
        double theta0 = std::sqrt(1.0/(term1 + term2 + term3));

        // Set value
        value = (theta <= theta0) ? m_norm : 0.0;

        // Compile option: Check for NaN/Inf
        #if defined(G_NAN_CHECK)
        if (gammalib::is_notanumber(value) || gammalib::is_infinite(value)) {
            std::cout << "*** ERROR: GModelSpatialEllipticalDisk::eval";
            std::cout << "(theta=" << theta << "): NaN/Inf encountered";
            std::cout << "(posangle=" << posangle << "): NaN/Inf encountered";
            std::cout << " (value=" << value;
            std::cout << ", m_term1=" << m_term1;
            std::cout << ", m_term2=" << m_term2;
            std::cout << ", m_term3=" << m_term3;
            std::cout << ", m_norm=" << m_norm;
            std::cout << ")" << std::endl;
        }
        #endif

    } // endif: position was inside enclosing circle

    // Return value
    return value;
}


/***********************************************************************//**
 * @brief Returns MC sky direction
 *
 * @param[in] energy Photon energy.
 * @param[in] time Photon arrival time.
 * @param[in,out] ran Random number generator.
 * @return Sky direction.
 *
 * Draws an arbitrary sky position from the 2D disk distribution.
 *
 * @todo Test function
 ***************************************************************************/
GSkyDir GModelSpatialEllipticalDisk::mc(const GEnergy& energy,
                                        const GTime&   time,
                                        GRan&          ran) const
{
    // Update precomputation cache
    update();

    // Initialise photon
    GPhoton photon;
    photon.energy(energy);
    photon.time(time);

    // Draw randomly from the radial disk and reject the value if its
    //outside the ellipse
    do {

        // Simulate offset from photon arrival direction
        double cosrad = std::cos(semimajor() * gammalib::deg2rad);
        double theta  = std::acos(1.0 - ran.uniform() * (1.0 - cosrad)) *
                        gammalib::rad2deg;
        double phi    = 360.0 * ran.uniform();

        // Rotate sky direction by offset
        GSkyDir sky_dir = dir();
        sky_dir.rotate_deg(phi, theta);

        // Set photon sky direction
        photon.dir(sky_dir);

    } while(GModelSpatialElliptical::eval(photon) <= 0.0);

    // Return photon direction
    return (photon.dir());
}


/***********************************************************************//**
 * @brief Checks where model contains specified sky direction
 *
 * @param[in] dir Sky direction.
 * @param[in] margin Margin to be added to sky direction (degrees)
 * @return True if the model contains the sky direction.
 *
 * Signals whether a sky direction is contained in the elliptical disk
 * model.
 *
 * @todo Implement correct evaluation of effective ellipse radius.
 ***************************************************************************/
bool GModelSpatialEllipticalDisk::contains(const GSkyDir& dir,
                                           const double&  margin) const
{
    // Compute distance to centre (radian)
    double distance = dir.dist(this->dir());

    // Return flag
    return (distance <= theta_max() + margin*gammalib::deg2rad);
}


/***********************************************************************//**
 * @brief Return maximum model radius (in radians)
 *
 * @return Returns maximum model radius.
 ***************************************************************************/
double GModelSpatialEllipticalDisk::theta_max(void) const
{
    // Set maximum model radius
    double theta_max = (semimajor() > semiminor())
                       ? semimajor() * gammalib::deg2rad
                       : semiminor() * gammalib::deg2rad;

    // Return value
    return theta_max;
}


/***********************************************************************//**
 * @brief Read model from XML element
 *
 * @param[in] xml XML element.
 *
 * @exception GException::model_invalid_parnum
 *            Invalid number of model parameters found in XML element.
 * @exception GException::model_invalid_parnames
 *            Invalid model parameter names found in XML element.
 *
 * Reads the elliptical disk model information from an XML element. The XML
 * element shall have either the format 
 *
 *     <spatialModel type="EllipticalDisk">
 *       <parameter name="RA"          scale="1.0" value="83.6331" min="-360" max="360" free="1"/>
 *       <parameter name="DEC"         scale="1.0" value="22.0145" min="-90"  max="90"  free="1"/>
 *       <parameter name="PA"          scale="1.0" value="45.0"    min="-360"  max="360" free="1"/>
 *       <parameter name="MinorRadius" scale="1.0" value="0.5"     min="0.001" max="10"  free="1"/>
 *       <parameter name="MajorRadius" scale="1.0" value="2.0"     min="0.001" max="10"  free="1"/>
 *     </spatialModel>
 *
 * or
 *
 *     <spatialModel type="EllipticalDisk">
 *       <parameter name="GLON"        scale="1.0" value="83.6331" min="-360" max="360" free="1"/>
 *       <parameter name="GLAT"        scale="1.0" value="22.0145" min="-90"  max="90"  free="1"/>
 *       <parameter name="PA"          scale="1.0" value="45.0"    min="-360"  max="360" free="1"/>
 *       <parameter name="MinorRadius" scale="1.0" value="0.5"     min="0.001" max="10"  free="1"/>
 *       <parameter name="MajorRadius" scale="1.0" value="2.0"     min="0.001" max="10"  free="1"/>
 *     </spatialModel>
 *
 * @todo Implement a test of the ellipse boundary. The axes
 *       and axes minimum should be >0.
 ***************************************************************************/
void GModelSpatialEllipticalDisk::read(const GXmlElement& xml)
{
    // Verify number of model parameters
    gammalib::xml_check_parnum(G_READ, xml, 5);

    // Read disk location
    GModelSpatialElliptical::read(xml);

    // Get parameters
    const GXmlElement* minor = gammalib::xml_get_par(G_READ, xml, m_semiminor.name());
    const GXmlElement* major = gammalib::xml_get_par(G_READ, xml, m_semimajor.name());

    // Read parameters
    m_semiminor.read(*minor);
    m_semimajor.read(*major);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Write model into XML element
 *
 * @param[in] xml XML element into which model information is written.
 *
 * Write the elliptical disk model information into an XML element. The XML
 * element will have the format 
 *
 *     <spatialModel type="EllipticalDisk">
 *       <parameter name="RA"          scale="1.0" value="83.6331" min="-360" max="360" free="1"/>
 *       <parameter name="DEC"         scale="1.0" value="22.0145" min="-90"  max="90"  free="1"/>
 *       <parameter name="PA"          scale="1.0" value="45.0"    min="-360"  max="360" free="1"/>
 *       <parameter name="MinorRadius" scale="1.0" value="0.5"     min="0.001" max="10"  free="1"/>
 *       <parameter name="MajorRadius" scale="1.0" value="2.0"     min="0.001" max="10"  free="1"/>
 *     </spatialModel>
 *
 ***************************************************************************/
void GModelSpatialEllipticalDisk::write(GXmlElement& xml) const
{
    // Verify model type
    gammalib::xml_check_type(G_WRITE, xml, type());

    // Write disk location
    GModelSpatialElliptical::write(xml);

    // Get or create parameters
    GXmlElement* minor = gammalib::xml_need_par(G_WRITE, xml, m_semiminor.name());
    GXmlElement* major = gammalib::xml_need_par(G_WRITE, xml, m_semimajor.name());

    // Write parameters
    m_semiminor.write(*minor);
    m_semimajor.write(*major);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Print information
 *
 * @param[in] chatter Chattiness.
 * @return String containing model information.
 ***************************************************************************/
std::string GModelSpatialEllipticalDisk::print(const GChatter& chatter) const
{
    // Initialise result string
    std::string result;

    // Continue only if chatter is not silent
    if (chatter != SILENT) {

        // Append header
        result.append("=== GModelSpatialEllipticalDisk ===");

        // Append parameters
        result.append("\n"+gammalib::parformat("Number of parameters"));
        result.append(gammalib::str(size()));
        for (int i = 0; i < size(); ++i) {
            result.append("\n"+m_pars[i]->print(chatter));
        }

    } // endif: chatter was not silent

    // Return result
    return result;
}


/*==========================================================================
 =                                                                         =
 =                            Private methods                              =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Initialise class members
 ***************************************************************************/
void GModelSpatialEllipticalDisk::init_members(void)
{
    // Initialise model type
    m_type = "EllipticalDisk";

    // Initialise precomputation cache. Note that zero values flag
    // uninitialised as a zero radius is not meaningful
    m_last_minor    = 0.0;
    m_last_major    = 0.0;
    m_minor_rad     = 0.0;
    m_major_rad     = 0.0;
    m_norm          = 0.0;
    m_last_posangle = 9999.0; // Signals that has not been initialised
    m_sin2pos       = 0.0;
    m_cospos2       = 0.0;
    m_sinpos2       = 0.0;
    m_minor2        = 0.0;
    m_major2        = 0.0;
    m_term1         = 0.0;
    m_term2         = 0.0;
    m_term3         = 0.0;

    // Return
    return;
}


/***********************************************************************//**
 * @brief Copy class members
 *
 * @param[in] model Elliptical disk model.
 *
 * We do not have to push back the members on the parameter stack as this
 * should have been done by init_members() that was called before. Otherwise
 * we would have the radius twice on the stack.
 ***************************************************************************/
void GModelSpatialEllipticalDisk::copy_members(const GModelSpatialEllipticalDisk& model)
{
    // Copy members
    m_type = model.m_type;   // Needed to conserve model type

    // Copy precomputation cache
    m_last_minor    = model.m_last_minor;
    m_last_major    = model.m_last_major;
    m_minor_rad     = model.m_minor_rad;
    m_major_rad     = model.m_major_rad;
    m_norm          = model.m_norm;
    m_last_posangle = model.m_last_posangle;
    m_sin2pos       = model.m_sin2pos;
    m_cospos2       = model.m_cospos2;
    m_sinpos2       = model.m_sinpos2;
    m_minor2        = model.m_minor2;
    m_major2        = model.m_major2;
    m_term1         = model.m_term1;
    m_term2         = model.m_term2;
    m_term3         = model.m_term3;

    // Return
    return;
}


/***********************************************************************//**
 * @brief Delete class members
 ***************************************************************************/
void GModelSpatialEllipticalDisk::free_members(void)
{
    // Return
    return;
}


/***********************************************************************//**
 * @brief Update precomputation cache
 *
 * Precomputes several variables in case that the model parameters
 * semiminor(), semimajor() or posangle() changed. Precomputation speeds
 * up the model evaluation.
 *
 * The method also computes the normalization
 *
 * \f[
 *    {\tt m\_norm} = \frac{1}{2 \pi \sqrt{(1-\cos a)(1-\cos b})}
 * \f]
 *
 * where
 * \f$a\f$ is the semimajor() axis and
 * \f$b\f$ is the semiminor() axis of the ellipse.
 *
 * @warning
 * The normalization of the elliptical disk is only valid in the small
 * angle approximation.
 ***************************************************************************/
void GModelSpatialEllipticalDisk::update() const
{
    // Initialise flag if something has changed
    bool changed = false;

    // Update if one axis has changed
    if (m_last_minor != semiminor() || m_last_major != semimajor()) {

        // Signal parameter changes
        changed = true;

        // Store last values
        m_last_minor = semiminor();
        m_last_major = semimajor();

        // Compute axes in radians
        m_minor_rad = m_last_minor * gammalib::deg2rad;
        m_major_rad = m_last_major * gammalib::deg2rad;

        // Take their squares
        m_minor2 = m_minor_rad * m_minor_rad;
        m_major2 = m_major_rad * m_major_rad;

        // Compute normalisation
        double denom = gammalib::twopi *
                       std::sqrt((1.0 - std::cos(m_minor_rad)) *
                                 (1.0 - std::cos(m_major_rad)));
        m_norm       = (denom > 0.0) ? 1.0 / denom : 0.0;

    } // endif: update required

    // Update chache if position angle changed
    if (m_last_posangle != posangle()) {

        // Signal parameter changes
        changed = true;

        // Store last value
        m_last_posangle = posangle();

        // Compute angle in radians
        double posangle_rad = m_last_posangle * gammalib::deg2rad;

        // Compute sine and cosine
        double cospos = std::cos(posangle_rad);
        double sinpos = std::sin(posangle_rad);

        // Cache important values for further computations
        m_cospos2 = cospos * cospos;
        m_sinpos2 = sinpos * sinpos;
        m_sin2pos = std::sin(2.0 * posangle_rad);

    } // endif: position angle update required

    // Perform precomputations in case anything has changed
    if (changed) {

        // Compute terms needed to compute inverse effective radius
        // squared
        m_term1 = m_sinpos2 / m_minor2 + m_cospos2 / m_major2;
        m_term2 = m_sinpos2 / m_major2 + m_cospos2 / m_minor2;
        m_term3 = m_sin2pos / m_major2 - m_sin2pos / m_minor2;

    } // endif: something has changed

    // Return
    return;
}


/***********************************************************************//**
 * @brief Set boundary sky region
 ***************************************************************************/
void GModelSpatialEllipticalDisk::set_region(void) const
{
    // Set maximum model radius
    double max_radius = (semimajor() > semiminor()) ? semimajor() : semiminor();

    // Set sky region circle
    GSkyRegionCircle region(dir(), max_radius);

    // Set region (circumvent const correctness)
    const_cast<GModelSpatialEllipticalDisk*>(this)->m_region = region;

    // Return
    return;
}
