/***************************************************************************
 *                  SPI helper classes for vector response                 *
 * ----------------------------------------------------------------------- *
 *  copyright (C) 2024 by Juergen Knoedlseder                              *
 * ----------------------------------------------------------------------- *
 *                                                                         *
 *  This program is free software: you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, either version 3 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 *                                                                         *
 ***************************************************************************/
/**
 * @file spi_helpers_response_vector.hpp
 * @brief Defintion of SPI helper classes for vector response
 * @author Juergen Knoedlseder
 */

#ifndef SPI_HELPERS_RESPONSE_VECTOR_HPP
#define SPI_HELPERS_RESPONSE_VECTOR_HPP

/* __ Includes ___________________________________________________________ */
#include "GFunctions.hpp"

/* __ Forward declarations _______________________________________________ */
class GSPIEventCube;
class GSPIReponse;
class GModelSpatialRadial;
class GVector;
class GMatrix;


/***********************************************************************//**
 * @class spi_radial_kerns_rho
 *
 * @brief Kernel for rho angle integration of radial models
 *
 * This class provides the kernel for the rho angle integration of radial
 * sky models.
 ***************************************************************************/
class spi_radial_kerns_rho : public GFunctions {
public:
    spi_radial_kerns_rho(const GSPIEventCube*       cube,
                         const GSPIResponse*        rsp,
                         const GModelSpatialRadial* radial,
                         const int&                 ipt,
                         const double*              livetimes,
                         const GMatrix&             rot,
                         const int&                 iter,
                         GVector&                   irfs) :
                         m_cube(cube),
                         m_rsp(rsp),
                         m_radial(radial),
                         m_ipt(ipt),
                         m_livetimes(livetimes),
                         m_rot(rot),
                         m_iter(iter),
                         m_irfs(irfs) { }
        int     size(void) const { return m_irfs.size(); }
        GVector eval(const double& omega);
    protected:
        const GSPIEventCube*       m_cube;            //!< Pointer to event cube
        const GSPIResponse*        m_rsp;             //!< Pointer to response
        const GModelSpatialRadial* m_radial;          //!< Radial spatial model
        const int&                 m_ipt;             //!< Pointing index
        const double*              m_livetimes;       //!< Livetime array
        const GMatrix&             m_rot;             //!< Rotation matrix
        const int&                 m_iter;            //!< Number of azimuthal iterations
        GVector&                   m_irfs;            //!< IRF vector to update
};


/***********************************************************************//**
 * @class spi_radial_kerns_omega
 *
 * @brief Kernel for azimuth angle integration of radial models
 *
 * This class provides the kernel for the azimuth angle integration of
 * radial sky models.
 ***************************************************************************/
class spi_radial_kerns_omega : public GFunctions {
public:
    spi_radial_kerns_omega(const GSPIEventCube*       cube,
                           const GSPIResponse*        rsp,
                           const int&                 ipt,
                           const double*              livetimes,
                           const GMatrix&             rot,
                           const double&              sin_rho,
                           const double&              cos_rho,
                           GVector&                   irfs) :
                           m_cube(cube),
                           m_rsp(rsp),
                           m_ipt(ipt),
                           m_livetimes(livetimes),
                           m_rot(rot),
                           m_sin_rho(sin_rho),
                           m_cos_rho(cos_rho),
                           m_irfs(irfs) { }
    int     size(void) const { return m_irfs.size(); }
    GVector eval(const double& omega);
protected:
    const GSPIEventCube*       m_cube;            //!< Pointer to event cube
    const GSPIResponse*        m_rsp;             //!< Pointer to response
    const int&                 m_ipt;             //!< Pointing index
    const double*              m_livetimes;       //!< Livetime array
    const GMatrix&             m_rot;             //!< Rotation matrix
    const double&              m_sin_rho;         //!< Sine of Rho
    const double&              m_cos_rho;         //!< Cosine of Rho
    GVector&                   m_irfs;            //!< IRF vector to update
};


/***********************************************************************//**
 * @class spi_elliptical_kerns_rho
 *
 * @brief Kernel for rho angle integration of elliptical models
 *
 * This class provides the kernel for the rho angle integration of elliptical
 * sky models.
 ***************************************************************************/
class spi_elliptical_kerns_rho : public GFunctions {
public:
    spi_elliptical_kerns_rho(const GSPIEventCube*           cube,
                             const GSPIResponse*            rsp,
                             const GModelSpatialElliptical* elliptical,
                             const int&                     ipt,
                             const double*                  livetimes,
                             const GMatrix&                 rot,
                             const int&                     iter,
                             GVector&                       irfs) :
                             m_cube(cube),
                             m_rsp(rsp),
                             m_elliptical(elliptical),
                             m_ipt(ipt),
                             m_livetimes(livetimes),
                             m_rot(rot),
                             m_iter(iter),
                             m_irfs(irfs) { }
        int     size(void) const { return m_irfs.size(); }
        GVector eval(const double& omega);
    protected:
        const GSPIEventCube*           m_cube;       //!< Pointer to event cube
        const GSPIResponse*            m_rsp;        //!< Pointer to response
        const GModelSpatialElliptical* m_elliptical; //!< Elliptical spatial model
        const int&                     m_ipt;        //!< Pointing index
        const double*                  m_livetimes;  //!< Livetime array
        const GMatrix&                 m_rot;        //!< Rotation matrix
        const int&                     m_iter;       //!< Number of azimuthal iterations
        GVector&                       m_irfs;       //!< IRF vector to update
};


/***********************************************************************//**
 * @class spi_elliptical_kerns_omega
 *
 * @brief Kernel for azimuth angle integration of elliptical models
 *
 * This class provides the kernel for the azimuth angle integration of
 * elliptical sky models.
 ***************************************************************************/
class spi_elliptical_kerns_omega : public GFunctions {
public:
    spi_elliptical_kerns_omega(const GSPIEventCube*           cube,
                               const GSPIResponse*            rsp,
                               const GModelSpatialElliptical* elliptical,
                               const int&                     ipt,
                               const double*                  livetimes,
                               const GMatrix&                 rot,
                               const double&                  rho,
                               const double&                  sin_rho,
                               const double&                  cos_rho,
                               GVector&                       irfs) :
                               m_cube(cube),
                               m_rsp(rsp),
                               m_elliptical(elliptical),
                               m_ipt(ipt),
                               m_livetimes(livetimes),
                               m_rot(rot),
                               m_rho(rho),
                               m_sin_rho(sin_rho),
                               m_cos_rho(cos_rho),
                               m_irfs(irfs) { }
    int     size(void) const { return m_irfs.size(); }
    GVector eval(const double& omega);
protected:
    const GSPIEventCube*           m_cube;       //!< Pointer to event cube
    const GSPIResponse*            m_rsp;        //!< Pointer to response
    const GModelSpatialElliptical* m_elliptical; //!< Elliptical spatial model
    const int&                     m_ipt;        //!< Pointing index
    const double*                  m_livetimes;  //!< Livetime array
    const GMatrix&                 m_rot;        //!< Rotation matrix
    const double&                  m_rho;        //!< Rho
    const double&                  m_sin_rho;    //!< Sine of Rho
    const double&                  m_cos_rho;    //!< Cosine of Rho
    GVector&                       m_irfs;       //!< IRF vector to update
};

#endif /* SPI_HELPERS_RESPONSE_VECTOR_HPP */
