/***************************************************************************
 *         GSPIEventCube.i - INTEGRAL/SPI event bin container class        *
 * ----------------------------------------------------------------------- *
 *  copyright (C) 2020-2025 by Juergen Knoedlseder                         *
 * ----------------------------------------------------------------------- *
 *                                                                         *
 *  This program is free software: you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, either version 3 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 *                                                                         *
 ***************************************************************************/
/**
 * @file GSPIEventCube.i
 * @brief INTEGRAL/SPI event bin container class definition
 * @author Juergen Knoedlseder
 */
%{
/* Put headers and other declarations here that are needed for compilation */
#include "GSPIEventCube.hpp"
%}


/***********************************************************************//**
 * @class GSPIEventCube
 *
 * @brief INTEGRAL/SPI event bin container class
 ***************************************************************************/
class GSPIEventCube : public GEventCube {

public:
    // Constructors and destructors
    GSPIEventCube(void);
    explicit GSPIEventCube(const GFilename& filename);
    GSPIEventCube(const GSPIEventCube& cube);
    virtual ~GSPIEventCube(void);

    // Implemented pure virtual base class methods
    virtual void           clear(void);
    virtual GSPIEventCube* clone(void) const;
    virtual std::string    classname(void) const;
    virtual int            size(void) const;
    virtual int            dim(void) const;
    virtual int            naxis(const int& axis) const;
    virtual void           load(const GFilename& filename);
    virtual void           save(const GFilename& filename,
                                const bool&      clobber = false) const;
    virtual void           read(const GFits& file);
    virtual void           write(GFits& file) const;
    virtual double         number(void) const;

    // Other methods
    double             ontime(void) const;
    double             livetime(void) const;
    int                models(void) const;
    double             model_counts(const int& index) const;
    const std::string& model_name(const int& index) const;
    const std::string& ptid(const int& ipt) const;
    const GSPIInstDir& dir(const int& ipt, const int& idet) const;
    const GSkyDir&     spi_x(const int& ipt) const;
    const GSkyDir&     spi_z(const int& ipt) const;

    // Event attribute methods
    const double&      event_size(const int& ievent) const;
    const double&      event_model(const int& ievent, const int& imodel) const;
};


/***********************************************************************//**
 * @brief GSPIEventCube class extension
 ***************************************************************************/
%extend GSPIEventCube {
    GSPIEventBin* __getitem__(int index) {
        if (index >= 0 && index < self->size()) {
            return (*self)[index];
        }
        else {
            throw GException::out_of_range("__getitem__(int)",
                                           "Event bin",
                                           index, self->size());
        }
    }
    void __setitem__(int index, const GSPIEventBin& val) {
        if (index>=0 && index < self->size()) {
            *((*self)[index]) = val;
        }
        else {
            throw GException::out_of_range("__setitem__(int)",
                                           "Event bin",
                                           index, self->size());
        }
    }
    GSPIEventCube copy() {
        return (*self);
    }
%pythoncode {
    def __getstate__(self):
        state = (self.classname()) # TODO: Replace by appropriate class members
        return state
    def __setstate__(self, state):
        self.__init__()
}
};
