/***************************************************************************
 *                 GVectorSparse.i - Sparse vector class                   *
 * ----------------------------------------------------------------------- *
 *  copyright (C) 2024 by Juergen Knoedlseder                              *
 * ----------------------------------------------------------------------- *
 *                                                                         *
 *  This program is free software: you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, either version 3 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 *                                                                         *
 ***************************************************************************/
/**
 * @file GVectorSparse.i
 * @brief Sparse vector class Python interface definition
 * @author Juergen Knoedlseder
 */

%{
/* Put headers and other declarations here that are needed for compilation */
#include "GVectorSparse.hpp"
#include "GException.hpp"
%}


/***********************************************************************//**
 * @class GVectorSparse
 *
 * @brief Sparse vector class
 ***************************************************************************/
class GVectorSparse : public GBase {
public:
    // Constructors and destructors
    GVectorSparse(void);
    GVectorSparse(const int& num, const int& alloc = 0);
    GVectorSparse(const GVector& vector);
    GVectorSparse(const GVectorSparse& vector);
    virtual ~GVectorSparse(void);

    // Vector operators
    bool operator==(const GVectorSparse& vector) const;
    bool operator!=(const GVectorSparse& vector) const;

    // Sparse vector methods
    void           clear(void);
    GVectorSparse* clone(void) const;
    std::string    classname(void) const;
    const int&     size(void) const;
    int            elements(void) const;
    const int&     inx(const int& index) const;
    const double&  data(const int& index) const;
};


/***********************************************************************//**
 * @brief GVectorSparse class extension
 ***************************************************************************/
%extend GVectorSparse {
    double __getitem__(const int& index) {
        if (index >= 0 && index < self->size()) {
            return (*self)[index];
        }
        else {
            throw GException::out_of_range("__getitem__(int)",
                                           "Sparse vector element index",
                                           index, self->size());
        }
    }
    void __setitem__(const int& index, const double& val) {
        if (index>=0 && index < self->size()) {
            (*self)[index] = val;
        }
        else {
            throw GException::out_of_range("__setitem__(int)",
                                           "Sparse vector element index",
                                           index, self->size());
        }
    }
    int __is__(const GVectorSparse &a) {
            return (*self) == a;
    }
    GVectorSparse copy() {
        return (*self);
    }
%pythoncode {
    def __getstate__(self):
        state = tuple([x for x in self]),
        return state
    def __setstate__(self, state):
        size = len(state[0])
        self.__init__(size)
        for i, x in enumerate(state[0]):
            self[i] = (x)
}
};
