/***************************************************************************
 *          GSPIModelDataSpace.cpp - INTEGRAL/SPI data space model         *
 * ----------------------------------------------------------------------- *
 *  copyright (C) 2020-2025 by Juergen Knoedlseder                         *
 * ----------------------------------------------------------------------- *
 *                                                                         *
 *  This program is free software: you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, either version 3 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 *                                                                         *
 ***************************************************************************/
/**
 * @file GSPIModelDataSpace.cpp
 * @brief INTEGRAL/SPI data space model implementation
 * @author Juergen Knoedlseder
 */

/* __ Includes ___________________________________________________________ */
#ifdef HAVE_CONFIG_H
#include <config.h>
#endif
#include <typeinfo>
#include "GException.hpp"
#include "GTools.hpp"
#include "GTimes.hpp"
#include "GVector.hpp"
#include "GMatrixSparse.hpp"
#include "GModelPar.hpp"
#include "GModelRegistry.hpp"
#include "GSPIModelDataSpace.hpp"
#include "GSPIObservation.hpp"
#include "GSPIEventBin.hpp"
#include "GSPITools.hpp"

/* __ Constants __________________________________________________________ */

/* __ Globals ____________________________________________________________ */
const GSPIModelDataSpace g_spi_data_space_seed;
const GModelRegistry     g_spi_data_space_registry(&g_spi_data_space_seed);

/* __ Method name definitions ____________________________________________ */
#define G_CONSTRUCTOR               "GSPIModelDataSpace::GSPIModelDataSpace("\
                        "GSPIObservation&, std::string&, std::string&, int&)"
#define G_EVAL1     "GSPIModelDataSpace::eval(GEvent&, GObservation&, bool&)"
#define G_EVAL2     "GSPIModelDataSpace::eval(GObservation&, GMatrixSparse*)"
#define G_NPRED  "GSPIModelDataSpace::npred(GEnergy&, GTime&, GObservation&)"
#define G_MC                   "GSPIModelDataSpace::mc(GObservation&, GRan&)"
#define G_SETUP                     "GSPIModelDataSpace::setup(Observation&)"
#define G_READ                       "GSPIModelDataSpace::read(GXmlElement&)"
#define G_WRITE                     "GSPIModelDataSpace::write(GXmlElement&)"
#define G_GET_DATE_TIME     "GSPIModelDataSpace::get_date_time(std::string&)"

/* __ Macros _____________________________________________________________ */

/* __ Coding definitions _________________________________________________ */

/* __ Debug definitions __________________________________________________ */
//#define G_DUMP_MC                                  //!< Dump MC information
//#define G_DEBUG_SETUP                               //!< Debug setup method
//#define G_DEBUG_SETUP_MODEL                       //!< Debug setup of model
//#define G_DEBUG_SETUP_MODEL_MAP           //!< Debug setup of parameter map
//#define G_DEBUG_SETUP_MODEL_FIX             //!< Debug fixing of parameters
//#define G_DEBUG_SPLIT_POINTING             //!< Debug splitting of pointing

/*==========================================================================
 =                                                                         =
 =                        Constructors/destructors                         =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Void constructor
 *
 * Constructs an empty INTEGRAL/SPI data space model.
 ***************************************************************************/
GSPIModelDataSpace::GSPIModelDataSpace(void) : GModelData()
{
    // Initialise members
    init_members();

    // Return
    return;
}


/***********************************************************************//**
 * @brief Model and method constructor
 *
 * @param[in] obs INTEGRAL/SPI observation.
 * @param[in] name Model name.
 * @param[in] method Fit method.
 * @param[in] index Model index.
 *
 * Constructs an INTEGRAL/SPI data space model by setting the fit method
 * and the model index.
 *
 * The constructor uses the INTEGRAL/SPI observation to setup the model
 * parameters and the mapping of parameters to the data space.
 ***************************************************************************/
GSPIModelDataSpace::GSPIModelDataSpace(const GSPIObservation& obs,
                                       const std::string&     name,
                                       const std::string&     method,
                                       const int&             index) :
                    GModelData()
{
    // Initialise members
    init_members();

    // Set members
    m_name   = name;
    m_method = method;
    m_index  = index;

    // Setup model
    setup(obs);

    // Return
    return;
}


/***********************************************************************//**
 * @brief XML constructor
 *
 * @param[in] xml XML element.
 *
 * Constructs an INTEGRAL/SPI data space model from the information that is
 * found in an XML element. Please refer to the read() method to learn more
 * about the information that is expected in the XML element.
 ***************************************************************************/
GSPIModelDataSpace::GSPIModelDataSpace(const GXmlElement& xml) :
                    GModelData(xml)
{
    // Initialise members
    init_members();

    // Read XML
    read(xml);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Copy constructor
 *
 * @param[in] model INTEGRAL/SPI data space model.
 *
 * Constructs an INTEGRAL/SPI data space model by copying information from an
 * existing model. Note that the copy is a deep copy, so the original object
 * can be destroyed after the copy without any loss of information.
 ***************************************************************************/
GSPIModelDataSpace::GSPIModelDataSpace(const GSPIModelDataSpace& model) :
                    GModelData(model)
{
    // Initialise private members for clean destruction
    init_members();

    // Copy members
    copy_members(model);

    // Return
    return;
}


/***********************************************************************//**
 * @brief Destructor
 *
 * Destroys an INTEGRAL/SPI data space model.
 ***************************************************************************/
GSPIModelDataSpace::~GSPIModelDataSpace(void)
{
    // Free members
    free_members();

    // Return
    return;
}


/*==========================================================================
 =                                                                         =
 =                               Operators                                 =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Assignment operator
 *
 * @param[in] model INTEGRAL/SPI data space model.
 * @return INTEGRAL/SPI data space model
 *
 * Assigns the information from an INTEGRAL/SPI data space model to the
 * class instance. Note that a deep copy of the information is performed, so
 * the @p model instance can be destroyed after the assignment without any
 * loss of information.
 ***************************************************************************/
GSPIModelDataSpace& GSPIModelDataSpace::operator=(const GSPIModelDataSpace& model)
{
    // Execute only if object is not identical
    if (this != &model) {

        // Copy base class members
        this->GModelData::operator=(model);

        // Free members
        free_members();

        // Initialise members
        init_members();

        // Copy members
        copy_members(model);

    } // endif: object was not identical

    // Return
    return *this;
}


/*==========================================================================
 =                                                                         =
 =                            Public methods                               =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Clear instance
 *
 * Resets the object to a clean initial state. All information that resided
 * in the object will be lost.
 ***************************************************************************/
void GSPIModelDataSpace::clear(void)
{
    // Free class members (base and derived classes, derived class first)
    free_members();
    this->GModelData::free_members();
    this->GModel::free_members();

    // Initialise members
    this->GModel::init_members();
    this->GModelData::init_members();
    init_members();

    // Return
    return;
}


/***********************************************************************//**
 * @brief Clone instance
 *
 * @return Pointer to deep copy of INTEGRAL/SPI data space model.
 *
 * Clone an INTEGRAL/SPI data space model. Cloning performs a deep copy of
 * the information, so the original object can be destroyed after cloning
 * without any loss of information.
 ***************************************************************************/
GSPIModelDataSpace* GSPIModelDataSpace::clone(void) const
{
    return new GSPIModelDataSpace(*this);
}


/***********************************************************************//**
 * @brief Evaluate function
 *
 * @param[in] event Observed event.
 * @param[in] obs Observation.
 * @param[in] gradients Compute gradients?
 * @return Model value.
 *
 * @exception GException::invalid_argument
 *            Observation does not contain a SPI event cube
 *            Event is not an SPI event bin.
 *
 * Evaluates the value of the SPI data space model for a given event and a
 * given observation. If @p gradients is true then parameter gradients are
 * computed.
 ***************************************************************************/
double GSPIModelDataSpace::eval(const GEvent&       event,
                                const GObservation& obs,
                                const bool&         gradients) const
{
    // Reset parameter indices
    m_has_eval_inx = false;
    m_eval_inx.clear();

    // Get pointer to event cube
    const GSPIEventCube* cube = dynamic_cast<const GSPIEventCube*>(obs.events());
    if (cube == NULL) {
        std::string cls = std::string(typeid(obs.events()).name());
        std::string msg = "Events of type \""+cls+"\" are not an INTEGRAL/SPI "
                          "event cube. Please specify an INTEGRAL/SPI observation "
                          "with an event cube as argument.";
        throw GException::invalid_argument(G_EVAL1, msg);
    }

    // Extract INTEGRAL/SPI event bin
    const GSPIEventBin* bin = dynamic_cast<const GSPIEventBin*>(&event);
    if (bin == NULL) {
        std::string cls = std::string(typeid(&event).name());
        std::string msg = "Event of type \""+cls+"\" is  not an INTEGRAL/SPI "
                          "event bin. Please specify an INTEGRAL/SPI event "
                          "bin as argument.";
        throw GException::invalid_argument(G_EVAL1, msg);
    }

    // Setup model
    setup(obs);

    // Initialise value
    double value = 0.0;

    // Set flag that signals that the model index is valid
    bool valid_index = (m_index >= 0 && m_index < cube->models());

    // Continue only if there are parameters
    if (m_parameters.size() > 0) {

        // Get bin size
        double size = bin->size();

        // Continue only if bin size is positive
        if (size > 0.0) {

            // Get bin index in data space
            int index = bin->index();

            // Get parameter index
            int ipar = m_map[index];

            // Get relevant model parameter
            const GModelPar* par = &(m_parameters[ipar]);

            // Get model value divided by size (note that the size will be
            // multiplied-in later, hence here we need the model value divided
            // by size). If the model index is not valid then set the model
            // value to unity.
            value = (valid_index) ? bin->model(m_index) / size : 1.0;

            // Get model scale factor
            double scale = par->value();

            // Optionally compute gradients
            if (gradients) {

                // Compute partial derivative
                double grad = (par->is_free()) ? value * par->scale() : 0.0;

                // Set gradient
                par->factor_gradient(grad);

                // Signal that gradient for this parameter was set
                m_has_eval_inx = true;
                m_eval_inx.push_back(ipar);

            } // endif: gradient computation was requested

            // Compute scaled model value
            value *= scale;

            // Compile option: Check for NaN/Inf
            #if defined(G_NAN_CHECK)
            if (gammalib::is_notanumber(value) || gammalib::is_infinite(value)) {
                std::cout << "*** ERROR: GSPIModelDataSpace::eval";
                std::cout << "(index=" << index << "):";
                std::cout << " NaN/Inf encountered";
                std::cout << " (value=" << value;
                std::cout << ", scale=" << scale;
                std::cout << ")" << std::endl;
            }
            #endif

        } // endif: binsize was positive

    } // endif: model was initialised

    // Return
    return value;
}


/***********************************************************************//**
 * @brief Return model values and gradients
 *
 * @param[in] obs Observation.
 * @param[out] gradients Pointer to matrix of gradients.
 * @param[in] offset Column index of first matrix gradient.
 * @return Model values.
 *
 * @exception GException::invalid_argument
 *            Observation does not contain a SPI event cube
 *
 * Evaluates the model values and parameter gradients for all events in an
 * observation. Gradients are only returned if the @p gradients pointer is
 * not NULL.
 *
 * The matrix of gradients is a sparse matrix where the number of rows
 * corresponds to the number of events and the number of columns corresponds
 * to the number of model parameters (see GObservation::model() method).
 *
 * An exception is thrown if the dimension of the @p gradients matrix is not
 * compatible with the model and the observations.
 ***************************************************************************/
GVector GSPIModelDataSpace::eval(const GObservation& obs,
                                 GMatrixSparse*      gradients,
                                 const int&          offset) const
{
    // Setup model
    setup(obs);

    // Get pointer to event cube
    const GSPIEventCube* cube = dynamic_cast<const GSPIEventCube*>(obs.events());
    if (cube == NULL) {
        std::string cls = std::string(typeid(obs.events()).name());
        std::string msg = "Events of type \""+cls+"\" are not an INTEGRAL/SPI "
                          "event cube. Please specify an INTEGRAL/SPI observation "
                          "with an event cube as argument.";
        throw GException::invalid_argument(G_EVAL2, msg);
    }

    // Get number of model parameters and number of events
    int npars   = size();         // Number of model parameters
    int nevents = cube->size();   // Number of events in cube
    int nmodels = cube->models(); // Number of models in cube (sky + background)

    // Initialise gradients flag
    bool grad = ((gradients != NULL) && (npars > 0));

    // If gradients were requested then check matrix consistency
    if (grad) {
        if (gradients->columns() < npars+offset) {
            std::string msg = "Number of "+gammalib::str(gradients->columns())+
                              " columns in gradient matrix is smaller than "
                              "the number of "+gammalib::str(npars)+" parameters "
                              "in model plus the offset "+gammalib::str(offset)+
                              ". Please provide a compatible gradient matrix.";
            throw GException::invalid_argument(G_EVAL2, msg);
        }
        if (gradients->rows() != nevents) {
            std::string msg = "Number of "+gammalib::str(gradients->rows())+
                              " rows in gradient matrix differs from number "
                              "of "+gammalib::str(nevents)+" events in "
                              "observation. Please provide a compatible "
                              "gradient matrix.";
            throw GException::invalid_argument(G_EVAL2, msg);
        }
    }

    // Initialise temporary memory pointers for gradient computation
    double** tmp_values = NULL;
    int**    tmp_rows   = NULL;
    int*     tmp_number = NULL;
    int*     tmp_inx    = NULL;

    // If gradient computation is requested then allocate temporary memory
    // for quick column-wise matrix filling. Filling the matrix element by
    // element is very slow, hence we store the gradient information in
    // temporary memory that can then be filled column-wise in the sparse
    // gradient matrix.
    if (grad) {

        // Allocate temporary memory for each column that will store the
        // gradient information in compressed vectors.
        tmp_values = new double*[npars]; // Compressed values for all columns
        tmp_rows   = new int*[npars];    // Row indices of values for all columns
        tmp_number = new int[npars];     // Number of elements for all columns
        tmp_inx    = new int[npars];     // Number of current element indices for all columns

        // Initialise temporary memory
        for (int i = 0; i < npars; ++i) {
            tmp_values[i] = NULL;
            tmp_rows[i]   = NULL;
            tmp_number[i] = 0;
            tmp_inx[i]    = 0;
        }

        // Determine the number of compressed values for all columns
        for (int k = 0; k < nevents; ++k) {
            if (cube->event_size(k) > 0.0) {
                int ipar  = m_map[k]; // Parameter index for event
                tmp_number[ipar]++;   // Increment number of times the index is used
            }
        }

        // Allocate memory for compressed column arrays
        for (int i = 0; i < npars; ++i) {
            if (tmp_number[i] > 0) {
                tmp_values[i] = new double[tmp_number[i]];
                tmp_rows[i]   = new int[tmp_number[i]];
            }
        }

    } // endif: gradient computation was requested

    // Initialise values
    GVector values(nevents);

    // Set flag that signals that the index is valid
    bool valid_index = (m_index >= 0 && m_index < cube->models());

    // Continue only if there are paramters
    if (m_parameters.size() > 0) {

        // Signal all parameters that will have analytical gradients. These
        // are all parameters that are free and for which the model provides
        // analytical gradients.
        if (grad) {
            for (int i = 0; i < npars; ++i) {
                const GModelPar& par = (*this)[i];
                if (par.is_free() && par.has_grad()) {
                    obs.computed_gradient(*this, par);
                }
            }
        } // endif: gradients were requested

        // Loop over events
        for (int k = 0; k < nevents; ++k) {

            // Get event bin size
            double size = cube->event_size(k);

            // Continue only if event bin size is positive
            if (size > 0.0) {

                // Get parameter index that corresponds to event bin
                int ipar = m_map[k];

                // Get relevant model parameter
                const GModelPar* par = &(m_parameters[ipar]);

                // Get model value divided by size (note that the size will be
                // multiplied-in later, hence here we need the model value divided
                // by size). If the model index is not valid then set the model value
                // to unity.
                double value = (valid_index) ? cube->event_model(k, m_index) / size : 1.0;

                // Store scaled model value
                values[k] = value * par->value();

                // Optionally compute gradients
                if (grad) {

                    // Compute partial derivative
                    double gradient = (par->is_free()) ? value * par->scale() : 0.0;

                    // Store gradient in temporary memory
                    //(*gradients)(k, ipar) = gradient;
                    tmp_values[ipar][tmp_inx[ipar]] = gradient; // Value
                    tmp_rows[ipar][tmp_inx[ipar]]   = k;        // Row index
                    tmp_inx[ipar]++;                            // Increment index

                } // endif: gradient computation was requested

            } // endif: binsize was positive

        } // endfor: looped over events

    } // endif: model was initialised

    // If gradient was requested then fill temporary memory in sparse
    // matrix and free temporary memory for quick filling
    if (grad) {

        // Fill temporary memory in sparse matrix
        for (int i = 0; i < npars; ++i) {
            if (tmp_number[i] > 0) {
                gradients->column(i+offset, tmp_values[i], tmp_rows[i], tmp_number[i]);
            }
        }

        // Free values
        if (tmp_values != NULL) {
            for (int i = 0; i < npars; ++i) {
                if (tmp_values[i] != NULL) delete [] tmp_values[i];
            }
            delete [] tmp_values;
        }

        // Free row indices
        if (tmp_rows != NULL) {
            for (int i = 0; i < npars; ++i) {
                if (tmp_rows[i] != NULL)   delete [] tmp_rows[i];
            }
            delete [] tmp_rows;
        }

        // Free number of elements and current index
        if (tmp_number != NULL) delete [] tmp_number;
        if (tmp_inx    != NULL) delete [] tmp_inx;

    } // endif: gradient computation was requested

    // Return values
    return values;
}


/***********************************************************************//**
 * @brief Model setup hook
 *
 * @param[in] obs Observation.
 *
 * Setup the model for a given observation by calling the setup_pars()
 * method if a SPI event cube is found in @p obs.
 ***************************************************************************/
void GSPIModelDataSpace::setup(const GObservation& obs) const
{
    // Continue only if observation pointer differs
    if (&obs != m_obs) {

        // Debug: signal that pointer differs
        #if defined(G_DEBUG_SETUP)
        std::cout << "GSPIModelDataSpace::setup_model: ";
        std::cout << "observation pointer differs";
        std::cout << std::endl;
        #endif

        // Extract INTEGRAL/SPI observation
        const GSPIObservation* spi_obs = dynamic_cast<const GSPIObservation*>(&obs);
        if (spi_obs == NULL) {
            std::string cls = std::string(typeid(&obs).name());
            std::string msg = "Observation of type \""+cls+"\" is not an "
                              "INTEGRAL/SPI observation. Please specify an "
                              "INTEGRAL/SPI observation as argument.";
            throw GException::invalid_argument(G_SETUP, msg);
        }

        // Store observation pointer
        m_obs = const_cast<GSPIObservation*>(spi_obs);

        // Extract INTEGRAL/SPI event cube. Continue only if the cube is
        // valid
        GSPIEventCube* cube = dynamic_cast<GSPIEventCube*>(m_obs->events());
        if (cube != NULL) {

            // Continue only if the method string is not empty
            if (!m_method.empty()) {
                const_cast<GSPIModelDataSpace*>(this)->setup_pars(cube);
            }

            // Debug: signal that index is not valid
            #if defined(G_DEBUG_SETUP)
            else {
                std::cout << "GSPIModelDataSpace::setup: ";
                std::cout << "no method defined.";
                std::cout << std::endl;
            }
            #endif

        } // endif: cube was valid

        // Debug: signal that event cube was invalid
        #if defined(G_DEBUG_SETUP)
        else {
            std::cout << "GSPIModelDataSpace::setup: ";
            std::cout << "no valid event cube found in observation";
            std::cout << std::endl;
        }
        #endif

    } // endif: observation pointer differed

    // Return
    return;
}


/***********************************************************************//**
 * @brief Return spatially integrated data model
 *
 * @param[in] obsEng Measured event energy.
 * @param[in] obsTime Measured event time.
 * @param[in] obs Observation.
 *
 * @exception GException::feature_not_implemented
 *            Feature not implemented
 *
 * @todo Implement method.
 ***************************************************************************/
double GSPIModelDataSpace::npred(const GEnergy&      obsEng,
                                 const GTime&        obsTime,
                                 const GObservation& obs) const
{
    // Initialise result
    double npred = 0.0;

    // Throw exception signaling that feature is not yet implemented
    throw GException::feature_not_implemented(G_NPRED);

    // Return
    return npred;
}


/***********************************************************************//**
 * @brief Return simulated list of events
 *
 * @param[in] obs Observation.
 * @param[in] ran Random number generator.
 *
 * @exception GException::feature_not_implemented
 *            Feature not implemented
 *
 * @todo Implement method.
 ***************************************************************************/
GSPIEventCube* GSPIModelDataSpace::mc(const GObservation& obs,
                                      GRan&               ran) const
{
    // Initialise new event cube
    GSPIEventCube* cube = NULL;

    // Throw exception signaling that feature is not yet implemented
    throw GException::feature_not_implemented(G_MC);

    // Return
    return cube;
}


/***********************************************************************//**
 * @brief Read model from XML element
 *
 * @param[in] xml XML element.
 *
 * Read the INTEGRAL/SPI data space model from an XML element.
 *
 * The model is composed of a list of scale parameters. The following XML
 * file format is expected:
 *
 *     <source name="Background" type="DataSpace" method="orbit,dete" index="0" instrument="SPI">
 *       <parameter name="REV0019_DETID00"        .../>
 *       <parameter name="REV0019_DETID01"        .../>
 *       ...
 *     </source>
 ***************************************************************************/
void GSPIModelDataSpace::read(const GXmlElement& xml)
{
    // Clear instance
    clear();

    // Set model method and index
    m_method = xml.attribute("method");
    m_index  = gammalib::toint(xml.attribute("index"));

    // Get number of parameters from XML file
    int npars = xml.elements("parameter");

    // Loop over all parameters
    for (int i = 0; i < npars; ++i) {

        // Allocate parameter
        GModelPar parameter;

        // Get XML element
        const GXmlElement* par = xml.element("parameter", i);

        // Read parameter from XML element
        parameter.read(*par);

        // Signal that parameter has gradient
        parameter.has_grad(true);

        // Push parameter on list
        m_parameters.push_back(parameter);

    } // endfor: looped over parameters

    // Read model attributes
    read_attributes(xml);

    // Set pointers
    set_pointers();

    // Return
    return;
}


/***********************************************************************//**
 * @brief Write model into XML element
 *
 * @param[in] xml XML element.
 *
 * @exception GException::invalid_value
 *            Existing XML element is not of required type
 *            Invalid number of model parameters or nodes found in XML element.
 *
 * Write the INTEGRAL/SPI data space model into an XML element.
 *
 * The model is composed of a list of scale parameters. The following XML
 * file structure will be written:
 *
 *     <source name="Background" type="DataSpace" method="orbit,dete" index="0" instrument="SPI">
 *       <parameter name="REV0019_DETID00"        .../>
 *       <parameter name="REV0019_DETID01"        .../>
 *       ...
 *     </source>
 ***************************************************************************/
void GSPIModelDataSpace::write(GXmlElement& xml) const
{
    // Initialise pointer on source
    GXmlElement* src = NULL;

    // Search corresponding source
    int n = xml.elements("source");
    for (int k = 0; k < n; ++k) {
        GXmlElement* element = xml.element("source", k);
        if (element->attribute("name") == name()) {
            src = element;
            break;
        }
    }

    // If no source with corresponding name was found then append one.
    // Set also the type and the instrument.
    if (src == NULL) {
        src = xml.append("source");
        src->attribute("name", name());
        src->attribute("type", type());
        if (instruments().length() > 0) {
            src->attribute("instrument", instruments());
        }
    }

    // Verify model type
    if (src->attribute("type") != type()) {
        std::string msg = "Invalid model type \""+src->attribute("type")+"\" "
                          "found in XML file. Model type \""+type()+"\" "
                          "expected.";
        throw GException::invalid_value(G_WRITE, msg);
    }

    // Determine number of parameters
    int npars = m_parameters.size();

    // If XML element has 0 parameters then append parameters
    if (src->elements() == 0) {
        for (int i = 0; i < npars; ++i) {
            src->append(GXmlElement("parameter"));
        }
    }

    // Verify that XML element has the required number of nodes
    if (src->elements() != npars || src->elements("parameter") != npars) {
        std::string msg = "Invalid number of parameters "+
                          gammalib::str(src->elements("parameter"))+
                          " found in XML file, but model requires exactly "+
                          gammalib::str(npars)+" parameters.";
        throw GException::invalid_value(G_WRITE, msg);
    }

    // Loop over all parameters
    for (int i = 0; i < npars; ++i) {

        // Get XML element
        GXmlElement* par = src->element("parameter", i);

        // Write parameter into XML element
        m_parameters[i].write(*par);

    } // endfor: looped over parameters

    // Write model attributes
    write_attributes(*src);

    // Write method and index
    src->attribute("method", m_method);
    src->attribute("index",  gammalib::str(m_index));

    // Return
    return;
}


/***********************************************************************//**
 * @brief Print model information
 *
 * @param[in] chatter Chattiness.
 * @return String containing model information.
 ***************************************************************************/
std::string GSPIModelDataSpace::print(const GChatter& chatter) const
{
    // Initialise result string
    std::string result;

    // Continue only if chatter is not silent
    if (chatter != SILENT) {

        // Append header
        result.append("=== GSPIModelDataSpace ===");

        // Append attributes
        result.append("\n"+print_attributes());

        // Append parameter summary
        result.append("\n"+gammalib::parformat("Method"));
        result.append(m_method);
        result.append("\n"+gammalib::parformat("Number of parameters"));
        result.append(gammalib::str(size()));
        for (int i = 0; i < size(); ++i) {
            result.append("\n"+m_pars[i]->print(chatter));
        }

    } // endif: chatter was not silent

    // Return result
    return result;
}


/*==========================================================================
 =                                                                         =
 =                            Private methods                              =
 =                                                                         =
 ==========================================================================*/

/***********************************************************************//**
 * @brief Initialise class members
 ***************************************************************************/
void GSPIModelDataSpace::init_members(void)
{
    // Initialise base class members
    m_instruments.clear();
    m_instruments.push_back("SPI");

    // Initialise members
    m_obs      = NULL;
    m_method.clear();
    m_index    = -1;
    m_map_size =  0;
    m_map      = NULL;
    m_parameters.clear();

    // Return
    return;
}


/***********************************************************************//**
 * @brief Copy class members
 *
 * @param[in] model Model.
 ***************************************************************************/
void GSPIModelDataSpace::copy_members(const GSPIModelDataSpace& model)
{
    // Copy members
    m_obs        = model.m_obs;       // Copy pointer
    m_method     = model.m_method;
    m_index      = model.m_index;
    m_map_size   = model.m_map_size;
    m_parameters = model.m_parameters;

    // Copy parameter map
    if (m_map_size > 0) {
        m_map = new int[m_map_size];
        for (int i = 0; i < m_map_size; ++i) {
            m_map[i] = model.m_map[i];
        }
    }

    // Set parameter pointers
    set_pointers();

    // Return
    return;
}


/***********************************************************************//**
 * @brief Delete class members
 ***************************************************************************/
void GSPIModelDataSpace::free_members(void)
{
    // Delete memory
    if (m_map != NULL) delete [] m_map;

    // Set pointers to free
    m_map = NULL;

    // Return
    return;
}


/***********************************************************************//**
 * @brief Set pointers
 *
 * Set pointers to all model parameters. The pointers are stored in a vector
 * that is member of the GModel base class.
 ***************************************************************************/
void GSPIModelDataSpace::set_pointers(void)
{
    // Clear parameter pointer(s)
    m_pars.clear();

    // Get number of parameters
    int npars = m_parameters.size();

    // Set pointers for all parameters
    for (int i = 0; i < npars; ++i) {
        m_pars.push_back(&(m_parameters[i]));
    }

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup parameters
 *
 * @param[in] cube INTEGRAL/SPI event cube
 *
 * Setup the parameters for a given model.
 *
 * We assume that the model index is within the valid range of possible
 * indices and that a method was specified.
 ***************************************************************************/
void GSPIModelDataSpace::setup_pars(GSPIEventCube* cube)
{
    // Free parameter map
    free_members();

    // Get cube dimensions
    int n_events = cube->size();
    int n_pt     = cube->naxis(0);
    int n_det    = cube->naxis(1);
    int n_eng    = cube->naxis(2);

    // Initialise index vectors for all three dimensions
    std::vector<int> pt_indices(n_pt,   0);
    std::vector<int> det_indices(n_det, 0);
    std::vector<int> eng_indices(n_eng, 0);

    // Initialise name vectors for all three dimensions
    std::vector<std::string> pt_names;
    std::vector<std::string> det_names;
    std::vector<std::string> eng_names;

    // Determine pointing indices
    setup_pointing_indices(cube, &pt_indices, &pt_names);

    // Determine detector indices
    setup_detector_indices(cube, &det_indices, &det_names);

    // Determine energy indices
    setup_energy_indices(cube, &eng_indices, &eng_names);

    // Determine number of parameters in each dimension. The number of
    // parameters is given by the last index in the vector + 1. The check
    // for a non-zero index vector size if just for security, if the event
    // cube is set it should always be positive, yet for an empty event cube
    // it may be zero, and in this case the method will just do nothing.
    int npt   = (pt_indices.size()  > 0) ? pt_indices[pt_indices.size()-1]   + 1 : 0;
    int ndet  = (det_indices.size() > 0) ? det_indices[det_indices.size()-1] + 1 : 0;
    int neng  = (eng_indices.size() > 0) ? eng_indices[eng_indices.size()-1] + 1 : 0;
    int npars = npt * ndet * neng;

    // Debug: print number of parameters in each dimension
    #if defined(G_DEBUG_SETUP_MODEL)
    std::cout << "GSPIModelDataSpace::setup_pars: ";
    std::cout << "determined pointing, detector and energy indices:" << std::endl;
    std::cout << "- npt = " << npt << std::endl;
    std::cout << "- ndet = " << ndet << std::endl;
    std::cout << "- neng = " << neng << std::endl;
    std::cout << "- npars = " << npars << std::endl;
    #endif

    // Continue only if there are parameters in the model
    if (npars > 0) {

        // Allocate parameters vector
        std::vector<GModelPar> parameters;
        parameters.reserve(npars);

        // Allocate parameters. Energy parameter indices vary the most fastest
        // followed by detector parameter indices, follwed by pointing parameter
        // indices.
        for (int ipt = 0; ipt < npt; ++ipt) {
            for (int idet = 0; idet < ndet; ++idet) {
                for (int ieng = 0; ieng < neng; ++ieng) {

                    // Build parameter name
                    std::string par_name;
                    if (m_name.empty()) {
                        par_name = "Scale";
                    }
                    else {
                        par_name = m_name;
                    }
                    if (eng_names.size() > 0) {
                        par_name += " " + eng_names[ieng];
                    }
                    if (det_names.size() > 0) {
                        par_name += " " + det_names[idet];
                    }
                    if (pt_names.size() > 0) {
                        par_name += " " + pt_names[ipt];
                    }

                    // If parameter name exists already in this class instance
                    // (because they were read using the read() method) then
                    // recover the parameter and push it into the vector of
                    // parameters
                    bool has_par = false;
                    for (int i = 0; i < m_parameters.size(); ++i) {
                        if (m_parameters[i].name() == par_name) {
                            parameters.push_back(m_parameters[i]);
                            has_par = true;
                            #if defined(G_DEBUG_SETUP_MODEL)
                            std::cout << "GSPIModelDataSpace::setup_pars: ";
                            std::cout << "copy old parameter \"";
                            std::cout << m_parameters[i].name() << "\"" << std::endl;
                            #endif
                            break;
                        }
                    }

                    // ... otherwise allocate new model parameter
                    if (!has_par) {
                        GModelPar par;
                        par.clear();
                        par.name(par_name);
                        par.free();
                        par.value(1.0);
                        par.scale(1.0);
                        par.gradient(0.0);
                        par.has_grad(true);
                        parameters.push_back(par);
                        #if defined(G_DEBUG_SETUP_MODEL)
                        std::cout << "GSPIModelDataSpace::setup_pars: ";
                        std::cout << "allocate new parameter \"";
                        std::cout << par_name << "\"" << std::endl;
                        #endif
                    }

                } // endfor: looped over energy parameters
            } // endfor: looped over detector parameters
        } // endfor: looped over pointing parameters

        // Allocate parameter map
        m_map_size = n_pt * n_det * n_eng;
        if (m_map_size > 0) {
            m_map = new int[m_map_size];
        }

        // Debug: print size of parameter map in each dimension
        #if defined(G_DEBUG_SETUP_MODEL)
        std::cout << "GSPIModelDataSpace::setup_pars: ";
        std::cout << "allocated parameter map:" << std::endl;
        std::cout << "- n_pt = " << n_pt << std::endl;
        std::cout << "- n_det = " << n_det << std::endl;
        std::cout << "- n_eng = " << n_eng << std::endl;
        std::cout << "- m_map_size = " << m_map_size << std::endl;
        #endif

        // Compute parameter map
        for (int i_pt = 0; i_pt < n_pt; ++i_pt) {
            for (int i_det = 0; i_det < n_det; ++i_det) {
                for (int i_eng = 0; i_eng < n_eng; ++i_eng) {

                    // Compute parameter index, respecting the scheme defined
                    // earlier: energy parameter indices vary the most fastest,
                    // followed by detector parameter indices, followed by
                    // pointing parameter indices
                    int index = pt_indices[i_pt]   * ndet * neng +
                                det_indices[i_det] * neng +
                                eng_indices[i_eng];

                    // Compute map index (same as in GSPIEventCube::read_dsp)
                    int i_map = (i_pt  * n_det + i_det) * n_eng + i_eng;

                    // Set map value
                    m_map[i_map] = index;

                    // Debug: print mapping
                    #if defined(G_DEBUG_SETUP_MODEL_MAP)
                    std::cout << "- [" << i_pt << "," << i_det << "," << i_eng << "]";
                    std::cout << " = " << index << std::endl;
                    #endif

                } // endfor: looped over energies
            } // endfor: looped over detectors
        } // endfor: looped over pointings

        // Replace model parameters
        m_parameters = parameters;

        // Set pointers
        set_pointers();

        // Get number of parameters
        int n_pars = m_parameters.size();

        // Determine number of events that use each parameter
        std::vector<int> n_uses(n_pars, 0);
        for (int i = 0; i < n_events; ++i) {
            if (cube->event_size(i) > 0.0) {
                int ipar = m_map[i]; // Parameter index for event bin
                n_uses[ipar]++;      // Increment usage of parameter
            }
        }

        // Fix all parameters that are not used
        for (int i = 0; i < n_pars; ++i) {
            if (n_uses[i] == 0) {
                m_parameters[i].fix();
                #if defined(G_DEBUG_SETUP_MODEL_FIX)
                std::cout << "GSPIModelDataSpace::setup_pars: ";
                std::cout << "parameter \"" << m_parameters[i].name() << "\" ";
                std::cout << "not used, fix parameter." << std::endl;
                #endif
            }
        }

    } // endif: the model has parameters

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup pointing indices
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of pointing indices.
 * @param[in,out] names Vector of pointing names.
 *
 * Setup vectors of pointing indices and pointing names. The length of the
 * vectors correspond to the number of pointings in the observation.
 ***************************************************************************/
void GSPIModelDataSpace::setup_pointing_indices(GSPIEventCube*            cube,
                                                std::vector<int>*         indices,
                                                std::vector<std::string>* names)
{
    // Convert method to lower case
    std::string method = gammalib::tolower(m_method);

    // Search for "point"
    if (gammalib::contains(method, "point")) {
        setup_point(cube, indices, names);
    }

    // Search for "orbit"
    else if (gammalib::contains(method, "orbit")) {
        setup_orbit(cube, indices, names);
    }

    // Search for "date"
    else if (gammalib::contains(method, "date")) {
        double time = get_date_time(method);
        setup_date(cube, indices, names, time);
    }

    // Handle "gedfail"
    if (gammalib::contains(method, "gedfail")) {
        add_gedfail(cube, indices, names);
    }

    // Handle "gedanneal"
    if (gammalib::contains(method, "gedanneal")) {
        add_gedanneal(cube, indices, names);
    }

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup detector indices
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of detector indices.
 * @param[in,out] names Vector of detector names.
 *
 * Setup vectors of detector indices and detector names. The length of the
 * vectors correspond to the number of detectors in the observation.
 ***************************************************************************/
void GSPIModelDataSpace::setup_detector_indices(GSPIEventCube*            cube,
                                                std::vector<int>*         indices,
                                                std::vector<std::string>* names)
{
    // Convert method to lower case
    std::string method = gammalib::tolower(m_method);

    // Search for "dete"
    if (gammalib::contains(method, "dete")) {
        setup_dete(cube, indices, names);
    }

    // Search for "evtclass"
    else if (gammalib::contains(method, "evtclass")) {
        setup_evtclass(cube, indices, names);
    }

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup energy indices
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of energy indices.
 * @param[in,out] names Vector of energy names.
 *
 * Setup vectors of energy indices and energy names. The length of the
 * vectors correspond to the number of energies in the observation.
 ***************************************************************************/
void GSPIModelDataSpace::setup_energy_indices(GSPIEventCube*            cube,
                                              std::vector<int>*         indices,
                                              std::vector<std::string>* names)
{
    // Convert method to lower case
    std::string method = gammalib::tolower(m_method);

    // Search for "ebin"
    if (gammalib::contains(method, "ebin")) {
        setup_ebin(cube, indices, names);
    }

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup pointing indices and names for "point" method
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of pointing indices.
 * @param[in,out] names Vector of pointing names.
 *
 * Setup vectors of pointing indices and names for "point" method.
 ***************************************************************************/
void GSPIModelDataSpace::setup_point(GSPIEventCube*            cube,
                                     std::vector<int>*         indices,
                                     std::vector<std::string>* names)
{
    // Get number of detectors
    int npt = indices->size();

    // Setup index vector
    for (int ipt = 0; ipt < npt; ++ipt) {
        (*indices)[ipt] = ipt;
        names->push_back("P" + gammalib::str(ipt, "%6.6d"));
    }

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup pointing indices and names for "orbit" method
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of pointing indices.
 * @param[in,out] names Vector of pointing names.
 *
 * Setup vectors of pointing indices and names for "orbit" method.
 ***************************************************************************/
void GSPIModelDataSpace::setup_orbit(GSPIEventCube*            cube,
                                     std::vector<int>*         indices,
                                     std::vector<std::string>* names)
{
    // Allocate orbit strings
    std::vector<std::string> orbits;

    // Get number of pointings
    int npt = indices->size();

    // Loop over all pointings
    for (int ipt = 0; ipt < npt; ++ipt) {

        // Initialise index
        int index = -1;

        // Get orbit
        std::string orbit = cube->ptid(ipt).substr(0,4);

        // If orbit is in orbits then store the location of the string
        // in the vector as parameter index.
        for (int i = 0; i < orbits.size(); ++i) {
            if (orbit == orbits[i]) {
                index = i;
                break;
            }
        }

        // If orbit is not yet in vector then
        if (index == -1) {
            index = orbits.size();
            orbits.push_back(orbit);
            names->push_back("O" + orbit);
        }

        // Store index
        (*indices)[ipt] = index;

    } // endfor: looped over pointings

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup pointing indices and names for "date" method
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of pointing indices.
 * @param[in,out] names Vector of pointing names.
 * @param[in] time Time scale in seconds.
 *
 * Setup vectors of pointing indices and names for "date" method.
 ***************************************************************************/
void GSPIModelDataSpace::setup_date(GSPIEventCube*            cube,
                                    std::vector<int>*         indices,
                                    std::vector<std::string>* names,
                                    const double&             time)
{
    // Compute start time, number of time bins and length of time bin in
    // seconds
    double tstart = cube->gti().tstart().secs();
    double tbin   = cube->gti().telapse();
    int    nbins  = int(tbin / time + 0.5);
    if (nbins < 1) {
        nbins = 1;
    }
    tbin /= double(nbins);

    // Get number of pointings
    int npt = indices->size();

    // Loop over all pointings
    for (int ipt = 0; ipt < npt; ++ipt) {

        // Get time of pointing in seconds
        double t = 0.5 * (cube->gti().tstart(ipt).secs() +
                          cube->gti().tstop(ipt).secs());

        // Compute grouping index. Make sure that it is in valid range.
        int index = int((t - tstart) / tbin);
        if (index < 0) {
            index = 0;
        }
        else if (index >= nbins) {
            index = nbins - 1;
        }

        // Store index
        (*indices)[ipt] = index;

    } // endfor: looped over indices

    // Remove unused indices
    for (int index = 0, used_index = 0; index < nbins; ++index) {

        // Replace all indices "index" by "used_index"
        int nindex = 0;
        for (int ipt = 0; ipt < npt; ++ipt) {
            if ((*indices)[ipt] == index) {
                (*indices)[ipt] = used_index;
                nindex++;
            }
        }

        // If we found indices to replace then add an index name and
        // increase "used_index"
        if (nindex > 0) {

            // Add date to names
            names->push_back("T" + gammalib::str(used_index, "%6.6d"));

            // Increment used index
            used_index++;

        } // endif: index was used

    } // endfor: looped over all pointings

    // Return
    return;
}


/***********************************************************************//**
 * @brief Modify pointing indices and names for "gedfail" method
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of pointing indices.
 * @param[in,out] names Vector of pointing names.
 *
 * Inserts additional model parameters for each pointing where a detector
 * failure occured.
 ***************************************************************************/
void GSPIModelDataSpace::add_gedfail(GSPIEventCube*            cube,
                                     std::vector<int>*         indices,
                                     std::vector<std::string>* names)
{
    // Set detector failure times
    GTimes times = gammalib::spi_gedfail_times();

    // Split pointings for all detector failures
    for (int i = 0; i < times.size(); ++i) {
        split_pointing_indices(cube, indices, names, times[i], "F"+gammalib::str(i));
    }

    // Return
    return;
}


/***********************************************************************//**
 * @brief Modify pointing indices and names for "gedanneal" method
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of pointing indices.
 * @param[in,out] names Vector of pointing names.
 *
 * Inserts additional model parameters for each pointing where a detector
 * annealing occured.
 ***************************************************************************/
void GSPIModelDataSpace::add_gedanneal(GSPIEventCube*            cube,
                                       std::vector<int>*         indices,
                                       std::vector<std::string>* names)
{
    // Set detector annealing start times
    GTimes times = gammalib::spi_annealing_start_times();

    // Split pointings for all detector annealings
    for (int i = 0; i < times.size(); ++i) {
        split_pointing_indices(cube, indices, names, times[i], "A"+gammalib::str(i, "%2.2d"));
    }

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup detector indices and names for "dete" method
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of detector indices.
 * @param[in,out] names Vector of detector names.
 *
 * Setup vectors of detector indices and names for "dete" method.
 ***************************************************************************/
void GSPIModelDataSpace::setup_dete(GSPIEventCube*            cube,
                                    std::vector<int>*         indices,
                                    std::vector<std::string>* names)
{
    // Get number of detectors
    int ndet = indices->size();

    // Setup index vector
    for (int idet = 0; idet < ndet; ++idet) {
        (*indices)[idet] = idet;
        names->push_back("D" + gammalib::str(idet, "%3.3d"));
    }

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup detector indices and names for "evtclass" method
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of detector indices.
 * @param[in,out] names Vector of detector names.
 *
 * Setup vectors of detector indices and names for "evtclass" method.
 ***************************************************************************/
void GSPIModelDataSpace::setup_evtclass(GSPIEventCube*            cube,
                                        std::vector<int>*         indices,
                                        std::vector<std::string>* names)
{
    // Get number of detectors
    int ndet = indices->size();

    // Flag which event classes occur in the actual dataset
    for (int idet = 0; idet < ndet; ++idet) {

        // Get detector ID for first pointing
        int detid = cube->dir(0, idet).detid();

        // Set the corresponding event class flag
        if (detid >=0 && detid < 19) {
            (*indices)[idet] = 0; // SE
        }
        else if (detid >= 19 && detid < 61) {
            (*indices)[idet] = 1; // ME2
        }
        else if (detid >= 61 && detid < 85) {
            (*indices)[idet] = 2; // ME3
        }
        else if (detid >= 85 && detid < 104) {
            (*indices)[idet] = 3; // PEE
        }
        else if (detid >= 104 && detid < 123) {
            (*indices)[idet] = 4; // PES
        }
        else if (detid >= 123 && detid < 142) {
            (*indices)[idet] = 5; // PEM
        }

    } // endfor: looped over all detectors

    // Remove unused indices
    for (int index = 0, used_index = 0; index < 6; ++index) {

        // Replace all indices "index" by "used_index"
        int nindex = 0;
        for (int idet = 0; idet < ndet; ++idet) {
            if ((*indices)[idet] == index) {
                (*indices)[idet] = used_index;
                nindex++;
            }
        }

        // If we found indices to replace then add an index name and
        // increase "used_index"
        if (nindex > 0) {

            // Add date to names
            names->push_back("C" + gammalib::str(used_index, "%1.1d"));

            // Increment used index
            used_index++;

        } // endif: index was used

    } // endfor: looped over all pointings

    // Return
    return;
}


/***********************************************************************//**
 * @brief Setup energy indices and names for "ebin" method
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of energy indices.
 * @param[in,out] names Vector of energy names.
 *
 * Setup vectors of energy indices and names for "ebin" method.
 ***************************************************************************/
void GSPIModelDataSpace::setup_ebin(GSPIEventCube*            cube,
                                    std::vector<int>*         indices,
                                    std::vector<std::string>* names)
{
    // Get number of detectors
    int neng = indices->size();

    // Setup index vector
    for (int ieng = 0; ieng < neng; ++ieng) {
        (*indices)[ieng] = ieng;
        names->push_back("E" + gammalib::str(ieng, "%3.3d"));
    }

    // Return
    return;
}


/***********************************************************************//**
 * @brief Get time scale from method string
 *
 * @param[in] method Method string.
 * @return Time scale (seconds).
 *
 * Get the time scale for the "DATE" method from the method string. The
 * following time units are supported: min, hour, day, week, month or year.
 ***************************************************************************/
double GSPIModelDataSpace::get_date_time(const std::string& method) const
{
    // Get position of "date" method in string
    size_t start = method.find("date") + 4;

    // Get scale of time
    double scale = 0.0;
    size_t stop  = std::string::npos;
    if ((stop = method.find("min", start)) != std::string::npos) {
        scale = 60.0;
    }
    else if ((stop = method.find("hour", start)) != std::string::npos) {
        scale = 3600.0;
    }
    else if ((stop = method.find("day", start)) != std::string::npos) {
        scale = 86400.0;
    }
    else if ((stop = method.find("week", start)) != std::string::npos) {
        scale = 604800.0;
    }
    else if ((stop = method.find("month", start)) != std::string::npos) {
        scale = 2628000.0;
    }
    else if ((stop = method.find("year", start)) != std::string::npos) {
        scale = 31536000.0;
    }

    // Throw an exception if no time scale was found
    if (stop == std::string::npos) {
        std::string msg = "Method string \""+method+"\" does not contain "
                          "time units for \"DATE\" method. Please specify one "
                          "of the following units: min, hour, day, week, "
                          "month or year.";
        throw GException::invalid_value(G_GET_DATE_TIME, msg);
    }

    // Get time in seconds
    double time = gammalib::todouble(method.substr(start, stop-start)) * scale;

    // Return time
    return time;
}


/***********************************************************************//**
 * @brief Split pointing indices and names at given time
 *
 * @param[in] cube Event cube.
 * @param[in,out] indices Vector of pointing indices.
 * @param[in,out] names Vector of pointing names.
 * @param[in] time Time of split.
 * @param[in] reason Reason for split.
 *
 * Split the pointing indices at a given time.
 ***************************************************************************/
void GSPIModelDataSpace::split_pointing_indices(GSPIEventCube*            cube,
                                                std::vector<int>*         indices,
                                                std::vector<std::string>* names,
                                                const GTime&              time,
                                                const std::string&        reason) const
{
    // Debug option: notify entry of method
    #if defined(G_DEBUG_SPLIT_POINTING)
    std::cout << "GSPIModelDataSpace::split_pointing_indices: ";
    std::cout << this->name() << " (" << m_method << ") at ";
    std::cout << time.utc() << " due to " << reason << std::endl;
    #endif

    // Get number of pointings
    int npt = indices->size();

    // Continue only if there are pointings
    if (npt > 0) {

        // Get index of largest pointing group plus one
        int igrp_last = 0;
        for (int ipt = 0; ipt < npt; ++ipt) {
            if ((*indices)[ipt] > igrp_last) {
                igrp_last = (*indices)[ipt];
            }
        }
        igrp_last++;

        // Initialise time span of pointing group
        int   ipt_start    = 0;
        int   ipt_stop     = 0;
        int   igrp_current = (*indices)[ipt_start];
        GTime tstart       = cube->gti().tstart(ipt_start);
        GTime tstop        = cube->gti().tstop(ipt_start);

        // Loop over all pointings
        for (int ipt = 0; ipt < npt; ++ipt) {

            // If pointing is in same pointing group then update stop
            // time
            if ((*indices)[ipt] == igrp_current) {
                tstop    = cube->gti().tstop(ipt);
                ipt_stop = ipt;
            }

            // If pointing is in different pointing group or the end of
            // the pointings has been reached then check whether specified
            // time is comprised in the time interval of the pointing
            // group. If this is the case an attempt is made to split
            // the pointing group into two.
            if (((*indices)[ipt] != igrp_current) || (ipt == npt-1)) {

                // If time is comprised within pointing group then split
                // pointing group
                if ((time > tstart) && (time < tstop)) {

                    // Assign a new pointing group index to all pointings
                    // that have a start time not earlier than the
                    // specified time
                    int nnew = 0;
                    for (int kpt = ipt_start; kpt <= ipt_stop; ++kpt) {
                        if (cube->gti().tstart(kpt) > time) {
                            (*indices)[kpt] = igrp_last;
                            nnew++;
                        }
                    }

                    // If a new pointing group was assigned then add a
                    // name to the vector and increment the new pointing
                    // group index. If no name exists so far then add an
                    // empty name since we split one pointing group into
                    // two.
                    if (nnew > 0) {
                        igrp_last++;
                        if (names->size() == 0) {
                            names->push_back("");
                        }
                        names->push_back(reason);

                        // Debug option: inform about insertion
                        #if defined(G_DEBUG_SPLIT_POINTING)
                        std::cout << "Insert new pointing group at pointing ";
                        std::cout << ipt << " (" << nnew << ")" << std::endl;
                        #endif
                    }

                } // endif: pointing group splitted

                // Update time span of next pointing group
                ipt_start    = ipt;
                ipt_stop     = ipt;
                igrp_current = (*indices)[ipt];
                tstart       = cube->gti().tstart(ipt);
                tstop        = cube->gti().tstop(ipt);

            } // endif: next pointing group or end encountered

        } // endfor: looped over pointings

    } // endif: there were pointings

    // Return
    return;
}
