/***************************************************************************
 *                  GVectorSparse.hpp - Sparse vector class                *
 * ----------------------------------------------------------------------- *
 *  copyright (C) 2024 by Juergen Knoedlseder                              *
 * ----------------------------------------------------------------------- *
 *                                                                         *
 *  This program is free software: you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, either version 3 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 *                                                                         *
 ***************************************************************************/
/**
 * @file GVectorSparse.hpp
 * @brief Sparce vector class interface definition
 * @author Juergen Knoedlseder
 */

#ifndef GSPARSEVECTOR_HPP
#define GSPARSEVECTOR_HPP

/* __ Includes ___________________________________________________________ */
#include <string>
#include "GBase.hpp"
#include "GException.hpp"

/* __ Definitions ________________________________________________________ */
#define G_SPARSE_VECTOR_DEFAULT_MEM_BLOCK   512  // Default Memory block size

/* __ Forward declarations _______________________________________________ */
class GVector;


/***********************************************************************//**
 * @class GVectorSparse
 *
 * @brief Sparse vector class
 *
 * This class implements a sparse double precision floating point vector
 * that is mainly used as an interface class to the sparse matrix class.
 * In that way, sparse matrix columns can be extracted and filled without
 * carying unnecessary zero elements.
 *
 * Note that GMatrixSparse is a friend class of GVectorSparse so that it
 * can directly manipulate the class members, avoiding any overhead.
 ***************************************************************************/
class GVectorSparse : public GBase {

    // Friend classes
    friend class GMatrixSparse;

public:
    // Constructors and destructors
    GVectorSparse(void);
    GVectorSparse(const int& num, const int& alloc = 0);
    GVectorSparse(const GVector& vector);
    GVectorSparse(const GVectorSparse& vector);
    virtual ~GVectorSparse(void);

    // Sparse vector element access operators
    double&       operator[](const int& index);
    const double& operator[](const int& index) const;

    // Vector operators
    bool           operator==(const GVectorSparse& vector) const;
    bool           operator!=(const GVectorSparse& vector) const;
    GVectorSparse& operator=(const GVectorSparse& vector);

    // Sparse vector methods
    void           clear(void);
    GVectorSparse* clone(void) const;
    std::string    classname(void) const;
    const int&     size(void) const;
    const int&     elements(void) const;
    const int&     inx(const int& index) const;
    const double&  data(const int& index) const;
    std::string    print(const GChatter& chatter = NORMAL) const;

private:
    // Private methods
    void init_members(void);
    void copy_members(const GVectorSparse& vector);
    void free_members(void);
    void alloc_members(const int& alloc);
    int  index2inx(const int& index) const;
    void insert(const int& index, const int& inx, const double& data);

    // Private data area
    int     m_num;      //!< Vector size
    int     m_elements; //!< Number of elements in vector
    int     m_alloc;    //!< Number of allocated elements
    int     m_colinx;   //!< Column index
    int*    m_inx;      //!< Index array
    double* m_data;     //!< Data array
};


/***********************************************************************//**
 * @brief Return class name
 *
 * @return String containing the class name ("GVectorSparse").
 ***************************************************************************/
inline
std::string GVectorSparse::classname(void) const
{
    return ("GVectorSparse");
}


/***********************************************************************//**
 * @brief Return full size of sparse vector
 *
 * @return Full size of sparse vector
 *
 * Returns the full size of the vector, including all zero elements. To
 * get the number of non-zero vector elements, use the elements() method.
 ***************************************************************************/
inline
const int& GVectorSparse::size() const
{
    return (m_num);
}


/***********************************************************************//**
 * @brief Return number of elements in sparse vector
 *
 * @return Number of elements in sparse vector
 *
 * Returns the number of non-zero vector elements.
 ***************************************************************************/
inline
const int& GVectorSparse::elements() const
{
    return (m_elements);
}


/***********************************************************************//**
 * @brief Return inx for sparse vector element
 *
 * @param[in] index Sparse vector element index.
 * @return Index of element in uncompressed vector.
 *
 * Returns the index of element @p index in uncompressed vector.
 *
 * This method does not perform any check on the validity of @p index, hence
 * the client needs to make sure that @p index is valid.
 ***************************************************************************/
inline
const int& GVectorSparse::inx(const int& index) const
{
    return (m_inx[index]);
}


/***********************************************************************//**
 * @brief Return value for sparse vector element
 *
 * @param[in] index Sparse vector element index.
 * @return Value of sparse vector element.
 *
 * Returns the value of element @p index.
 *
 * This method does not perform any check on the validity of @p index, hence
 * the client needs to make sure that @p index is valid.
 ***************************************************************************/
inline
const double& GVectorSparse::data(const int& index) const
{
    return (m_data[index]);
}

#endif /* GSPARSEVECTOR_HPP */
