/***************************************************************************
 *      GModelDataMultiplicative.hpp - Multiplicative data model class     *
 * ----------------------------------------------------------------------- *
 *  copyright (C) 2025 by Juergen Knoedlseder                              *
 * ----------------------------------------------------------------------- *
 *                                                                         *
 *  This program is free software: you can redistribute it and/or modify   *
 *  it under the terms of the GNU General Public License as published by   *
 *  the Free Software Foundation, either version 3 of the License, or      *
 *  (at your option) any later version.                                    *
 *                                                                         *
 *  This program is distributed in the hope that it will be useful,        *
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of         *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the          *
 *  GNU General Public License for more details.                           *
 *                                                                         *
 *  You should have received a copy of the GNU General Public License      *
 *  along with this program.  If not, see <http://www.gnu.org/licenses/>.  *
 *                                                                         *
 ***************************************************************************/
/**
 * @file GModelDataMultiplicative.hpp
 * @brief Multiplicative data model class interface definition
 * @author Juergen Knoedlseder
 */

#ifndef GMODELDATAMULTIPLICATIVE_HPP
#define GMODELDATAMULTIPLICATIVE_HPP

/* __ Includes ___________________________________________________________ */
#include <string>
#include <vector>
#include "GModelData.hpp"

/* __ Forward declarations _______________________________________________ */
class GRan;
class GEnergy;
class GTime;
class GEvent;
class GObservation;
class GXmlElement;


/***********************************************************************//**
 * @class GModelDataMultiplicative
 *
 * @brief Multiplicative data model class
 *
 * This class implements a multiplicative data model. The data model
 * components can be defined in an XML file, or added using the append()
 * method.
 ***************************************************************************/
class GModelDataMultiplicative : public GModelData {

public:
    // Constructors and destructors
    GModelDataMultiplicative(void);
    explicit GModelDataMultiplicative(const GXmlElement& xml);
    GModelDataMultiplicative(const GModelDataMultiplicative& model);
    virtual ~GModelDataMultiplicative(void);

    // Operators
    virtual GModelDataMultiplicative& operator=(const GModelDataMultiplicative& model);

    // Implemented pure virtual methods
    virtual void                      clear(void);
    virtual GModelDataMultiplicative* clone(void) const;
    virtual std::string               classname(void) const;
    virtual std::string               type(void) const;
    virtual bool                      is_constant(void) const;
    virtual double                    eval(const GEvent& event,
                                           const GObservation& obs,
                                           const bool& gradients = false) const;
    virtual double                    npred(const GEnergy& obsEng, const GTime& obsTime,
                                            const GObservation& obs) const;
    virtual GEvents*                  mc(const GObservation& obs, GRan& ran) const;
    virtual void                      read(const GXmlElement& xml);
    virtual void                      write(GXmlElement& xml) const;
    virtual std::string               print(const GChatter& chatter = NORMAL) const;

    // Implemented pure virtual base class methods
    virtual GVector eval(const GObservation& obs,
                         GMatrixSparse* gradients = NULL,
                         const int& offset = 0) const;

    // Other methods
    void              append(const GModelData& model);
    int               components(void) const;
    const GModelData* component(const int& index) const;
    const GModelData* component(const std::string& name) const;

protected:
    // Protected methods
    void init_members(void);
    void copy_members(const GModelDataMultiplicative& model);
    void free_members(void);

    // Protected members
    std::string              m_type;   //!< Model type
    std::vector<GModelData*> m_models; //!< Container of data models
};


/***********************************************************************//**
 * @brief Return class name
 *
 * @return String containing the class name ("GModelDataMultiplicative").
 ***************************************************************************/
inline
std::string GModelDataMultiplicative::classname(void) const
{
    return ("GModelDataMultiplicative");
}


/***********************************************************************//**
 * @brief Return model type
 *
 * @return Model type.
 *
 * Returns the type of the multiplicative data model.
 ***************************************************************************/
inline
std::string GModelDataMultiplicative::type(void) const
{
    return (m_type);
}


/***********************************************************************//**
 * @brief Return number of data model components
 *
 * @return Number of data model components.
 *
 * Returns the number of data model components.
 ***************************************************************************/
inline
int GModelDataMultiplicative::components(void) const
{
    return ((int)m_models.size());
}

#endif /* GMODELDATAMULTIPLICATIVE_HPP */
